package gui;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Properties;
import java.util.logging.Logger;

public class ApplicationConfigHandler
{
    public enum BoxType
    {
        SAT,
        KABEL
    };

    public enum Converter
    {
        KEINE_KONVERTIERUNG,
        AAC,
        MP3
    };

    private IApplicationConfig config = new ApplicationConfig();

    private static final Logger log = Logger.getLogger(ApplicationConfigHandler.class.getName());

    private boolean initialized = false;

    private File configFile = null;

    public ApplicationConfigHandler()
    {
        // Im Construktor dürfen noch keine Log-Ausgaben erfolgen, erst, wenn die GUI erstellt wurde!
    }

    private void initConfig()
    {
        String filename = System.getProperty("user.home") + File.separator + ".tsStreamRipper.properties";
        configFile = new File(filename);
        if (!configFile.exists())
        {
            log.info("Es existiert noch keine Config-Datei. Lege neue mit Default-Werten an ...");
            saveConfig();
        }
        try
        {
            loadConfig();
        }
        catch (IOException e)
        {
            log.warning("Konfiguration läßt sich nicht laden!");
            log.throwing("ApplicationConfig", "CONSTRUCTOR", e);
        }
    }

    public IApplicationConfig getConfig()
    {
        if (!initialized)
        {
            initialized = true;
            initConfig();
        }
        return config;
    }

    public void resetConfig()
    {
        if (!initialized)
        {
            initialized = true;
            initConfig();
        }
        try
        {
            loadConfig();
        }
        catch (IOException e)
        {
            log.warning("Konfiguration läßt sich nicht resetten!");
            log.throwing("ApplicationConfig", "resetConfig", e);
        }
    }

    private void loadConfig() throws IOException
    {
        log.fine("Lade Konfiguration ...");
        Properties props = new Properties();
        FileInputStream fis = null;
        try
        {
            fis = new FileInputStream(configFile);

            props.load(fis);
            config.setHost(props.getProperty("host"));
            config.setPort(props.getProperty("port", "8001"));
            config.setWebport(props.getProperty("webport", "80"));
            config.setUsername(props.getProperty("username"));
            config.setPasswort(props.getProperty("passwort"));
            config.setBoxType(BoxType.valueOf(props.getProperty("boxType", "SAT")));
            config.setTmpDir(props.getProperty("tmpDir"));
            config.setConverter(Converter.valueOf(props.getProperty("converter", "KEINE_KONVERTIERUNG")));
            config.setFaacPath(props.getProperty("faacPath"));
            config.setLamePath(props.getProperty("lamePath"));
            config.setOverwrite(Boolean.valueOf(props.getProperty("overwrite", "FALSE")));
            config.setVerbose(Boolean.valueOf(props.getProperty("verbose", "FALSE")));
            config.setSwitchFilename(Boolean.valueOf(props.getProperty("switchFilename", "FALSE")));
            config.setMusic60er70er(props.getProperty("music60er70er"));
            config.setMusic80er90er(props.getProperty("music80er90er"));
            config.setMusicCountry(props.getProperty("musicCountry"));
            config.setMusicDeutscheCharts(props.getProperty("musicDeutscheCharts"));
            config.setMusicLovesongs(props.getProperty("musicLovesongs"));
            config.setMusicRnbHiphop(props.getProperty("musicRnbHiphop"));
            config.setMusicRockHymnen(props.getProperty("musicRockHymnen"));
            config.setMusicFreieAuswahl(props.getProperty("musicFreieAuswahl"));
            config.setRestdauerKorrektur(props.getProperty("restdauerKorrektur"));
            log.fine("Konfiguration geladen.");
        }
        finally
        {
            if (fis != null)
            {
                fis.close();
            }
        }
    }

    private String getNullableString(String s)
    {
        return s == null ? "" : s;
    }

    private String getDefaultString(String s, String defaultValue)
    {
        return s == null ? defaultValue : s;
    }

    public void saveConfig()
    {
        if (!initialized)
        {
            initialized = true;
            initConfig();
        }

        log.fine("Speichere Konfiguration ...");

        FileOutputStream fos = null;
        try
        {
            fos = new FileOutputStream(configFile);
            Properties props = new Properties();
            props.put("host", getNullableString(config.getHost()));
            props.put("port", getNullableString(config.getPort()));
            props.put("webport", getNullableString(config.getWebport()));
            props.put("username", getNullableString(config.getUsername()));
            props.put("passwort", getNullableString(config.getPasswort()));
            props.put("boxType", getNullableString(config.getBoxType().toString()));
            props.put("tmpDir", getNullableString(config.getTmpDir()));
            props.put("converter", getNullableString(config.getConverter().toString()));
            props.put("faacPath", getNullableString(config.getFaacPath()));
            props.put("lamePath", getNullableString(config.getLamePath()));
            props.put("overwrite", getNullableString(Boolean.toString(config.isOverwrite())));
            props.put("verbose", getNullableString(Boolean.toString(config.isVerbose())));
            props.put("switchFilename", getNullableString(Boolean.toString(config.isSwitchFilename())));
            props.put("music60er70er", getNullableString(config.getMusic60er70er()));
            props.put("music80er90er", getNullableString(config.getMusic80er90er()));
            props.put("musicCountry", getNullableString(config.getMusicCountry()));
            props.put("musicDeutscheCharts", getNullableString(config.getMusicDeutscheCharts()));
            props.put("musicLovesongs", getNullableString(config.getMusicLovesongs()));
            props.put("musicRnbHiphop", getNullableString(config.getMusicRnbHiphop()));
            props.put("musicRockHymnen", getNullableString(config.getMusicRockHymnen()));
            props.put("musicFreieAuswahl", getNullableString(config.getMusicFreieAuswahl()));
            props.put("restdauerKorrektur", getDefaultString(config.getRestdauerKorrektur(), "20"));
            

            props.store(fos, "tsStreamRipper Konfiguration");
        }
        catch (IOException e)
        {
            log.warning("Konfiguration läßt sich nicht speichern!");
            log.throwing("ApplicationConfig", "saveConfig", e);
        }
        finally
        {
            try
            {
                fos.close();
            }
            catch (IOException e)
            {
                e.printStackTrace();
            }
        }
    }

}
