package gui;

import gui.ApplicationConfigHandler.BoxType;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Font;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.util.Properties;
import java.util.ResourceBundle;
import java.util.logging.Logger;

import javax.swing.GroupLayout;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.LayoutStyle;
import javax.swing.SwingConstants;
import javax.swing.UIManager;
import javax.swing.WindowConstants;

import tsStreamRipper.Configuration;
import tsStreamRipper.StreamIt;
import tsStreamRipper.audio.ConverterOption;
import tsStreamRipper.audio.ConverterFactory.ConverterType;
import tsStreamRipper.dream.filestream.IFileStreamNameFormatter;
import tsStreamRipper.dream.filestream.InterpretTitelFormat;
import tsStreamRipper.dream.filestream.TitelInterpretFormat;
import tsStreamRipper.dream.webinterface.HttpParameter;
import tsStreamRipper.dream.webinterface.parser.EPGType;
import tsStreamRipper.dream.webstreaming.IExceptionCallback;
import tsStreamRipper.dream.webstreaming.StreamChannel;
import tsStreamRipper.util.logging.Config;
import tsStreamRipper.util.logging.ILogEntryObserver;

/**
 * Hauptfenster
 */
public class MainWindow extends JFrame implements ILogEntryObserver, IExceptionCallback
{
    private static final Logger log = Logger.getLogger(MainWindow.class.getName());

    private ApplicationConfigHandler configHandler = new ApplicationConfigHandler();

    private Properties channelIdList = new Properties();

    private StreamChannel chan = null;

    private MessageDialog dlgMessage = new MessageDialog(this);

    /**
     * 1 MB Puffer für die Textausgabe (Log)
     */
    private static final int MAX_CHAR_LOG_BUFFER = 1000000;

    /**
     * 
     */
    private static final long serialVersionUID = 1L;

    public MainWindow()
    {
        initComponents();
        modusKeineAufnahme();
        reloadConfig();
    }

    private void modusKeineAufnahme()
    {
        lblAufnahmeLaeuft.setVisible(false);
        btnStream.setEnabled(true);
        btnStopp.setEnabled(false);
        //
        // Stream-Threads stoppen asynchron. Von daher müsste die GUI korrekt Benachrichtigt werden ...
        // Dieser Hack ist vermutlich aber ausreichend, weil der Status "Stream läuft noch" bisher nur eine Warnmeldung
        // erzeugt, falls Einstellungen geändert werden.
        //
        GuiManagerBean.getInstance().setStreamRunning(false);
    }

    private void modusAufnahme()
    {
        lblAufnahmeLaeuft.setVisible(true);
        btnStream.setEnabled(false);
        btnStopp.setEnabled(true);
        //
        // Stream-Threads stoppen asynchron. Von daher müsste die GUI korrekt Benachrichtigt werden ...
        // Dieser Hack ist vermutlich aber ausreichend, weil der Status "Stream läuft noch" bisher nur eine Warnmeldung
        // erzeugt, falls Einstellungen geändert werden.
        //
        GuiManagerBean.getInstance().setStreamRunning(true);
    }

    private boolean validateStreamData()
    {
        if (tfSref.getText().trim().equals(""))
        {
            dlgMessage.setErrorText("Bitte geben Sie eine Channel ID an.");
            dlgMessage.setVisible(true);
            return false;
        }
        if (tfMusicDir.getText().trim().equals(""))
        {
            dlgMessage.setErrorText("Bitte geben Sie ein Aufnahmeverzeichnis an.");
            dlgMessage.setVisible(true);
            return false;
        }

        File f = new File(tfMusicDir.getText());
        if (!f.isDirectory())
        {
            dlgMessage.setErrorText("Das angegebene Aufnahmeverzeichnis ist kein Verzeichnis!");
            dlgMessage.setVisible(true);
            return false;
        }

        // Host / Port
        IApplicationConfig c = configHandler.getConfig();
        if (c.getHost().trim().equals("") || c.getPort().trim().equals(""))
        {
            dlgMessage.setErrorText("Bitte gehen Sie in das Menü 'Einstellungen' und konfigurieren den Hostnamen und Port Ihrer Dreambox.");
            dlgMessage.setVisible(true);
            return false;
        }

        // Tmp Directory
        if (!"".equals(c.getTmpDir()))
        {
            f = new File(c.getTmpDir());
            if (!f.isDirectory())
            {
                dlgMessage.setErrorText("Das angegebene Temp-Verzeichnis ist kein Verzeichnis! Tmp-Pfad in Einstellungen korrekt angeben oder Eintrag löschen.");
                dlgMessage.setVisible(true);
                return false;
            }
        }

        return true;
    }

    private void reloadConfig()
    {
        configHandler.resetConfig();

        // Als erstes die Channel Proeprties laden, weil diese in Folge benötigt werden.
        try
        {
            if (configHandler.getConfig().getBoxType().equals(BoxType.SAT))
            {
                channelIdList.load(StreamIt.class.getResourceAsStream("/channel.properties"));
            }
            else if (configHandler.getConfig().getBoxType().equals(BoxType.KABEL))
            {
                channelIdList.load(StreamIt.class.getResourceAsStream("/channel_kd.properties"));
            }
            else
            {
                log.severe("Boxtyp unbekannt");
            }
        }
        catch (Exception e)
        {
            log.severe("Laden der Channel-Property Liste fehlgeschlagen.");
        }

        // Anzeigenamen Laden

        cbChannel.removeAllItems();
        cbChannel.addItem(ChannelEntry.FREI);
        cbChannel.addItem(ChannelEntry.CE_60er_70er);
        cbChannel.addItem(ChannelEntry.CE_80er_90er);
        cbChannel.addItem(ChannelEntry.CE_Country);
        cbChannel.addItem(ChannelEntry.CE_DeutscheCharts);
        cbChannel.addItem(ChannelEntry.CE_Lovesongs);
        cbChannel.addItem(ChannelEntry.CE_RnB_Hiphop);
        cbChannel.addItem(ChannelEntry.CE_RockHymnen);

        cbEpgType.removeAllItems();
        cbEpgType.addItem(EPGType.SKY_RADIO);
        cbEpgType.addItem(EPGType.KD_23_5_GRAD);
        cbEpgType.addItem(EPGType.SWISS_SATELLITE_RADIO);
    }

    private void startStreaming()
    {
        IApplicationConfig c = configHandler.getConfig();
        HttpParameter params = new HttpParameter(c.getHost(), Integer.parseInt(c.getWebport()), Integer.parseInt(c.getPort()), c.getUsername(), c.getPasswort());

        try
        {
            ConverterOption convertOpt = null;
            switch (c.getConverter())
            {
                case KEINE_KONVERTIERUNG:
                    break;
                case MP3:
                    convertOpt = new ConverterOption(ConverterType.MP3, new File(c.getLamePath()), c.isRemoveOriginalAfterConvert());
                    break;
                case AAC:
                    convertOpt = new ConverterOption(ConverterType.AAC, new File(c.getFaacPath()), c.isRemoveOriginalAfterConvert());
                    break;
                default:
                    break;
            }

            IFileStreamNameFormatter nameFormat = null;

            if (c.isSwitchFilename())
            {
                nameFormat = new InterpretTitelFormat();
            }
            else
            {
                // Default Title Format
                nameFormat = new TitelInterpretFormat();
            }

            File tmpDir = null;
            if (!c.getTmpDir().trim().equals(""))
            {
                tmpDir = new File(c.getTmpDir());
            }

            EPGType epgType = (EPGType) cbEpgType.getSelectedItem();
            Configuration config = new Configuration(params, tfSref.getText(), tmpDir, new File(tfMusicDir.getText()), convertOpt, nameFormat, c.isOverwrite(), Integer.parseInt(c.getRestdauerKorrektur()), epgType);
            chan = new StreamChannel(this, config);

            Runnable r = new Runnable()
            {
                public void run()
                {
                    chan.start();
                    try
                    {
                        chan.join();
                    }
                    catch (InterruptedException e)
                    {
                        e.printStackTrace();
                    }
                    modusKeineAufnahme();
                }
            };
            Thread t = new Thread(r);
            t.start();
        }
        catch (Exception e)
        {
            log.warning(e.getMessage());
        }
    }

    /**
     * GUI erstellen.
     */
    private static void createAndShowGUI()
    {
        log.finer("starte GUI ...");
        MainWindow mw = new MainWindow();
        mw.setVisible(true);
        GuiManagerBean.getInstance().setWindow(mw);

        log.info(StreamIt.version);
        log.info("tsStreamRipper GUI initialisiert.");
        log.info("Für die Kommandozeilen Hilfe das Programm mit 'java -jar tsStreamRipper.jar --help' starten.");
    }

    public static void showMainWindow()
    {
        // Look & Feel setzen ... Plattformspezifisch
        try
        {
            UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }

        // Gui-Manager initialisieren und starten ...
        GuiManagerBean.getInstance().setGuiRunning(true);

        javax.swing.SwingUtilities.invokeLater(new Runnable()
        {
            public void run()
            {
                Config.configureRuntimeLogging(true, true);
                createAndShowGUI();
            }
        });
    }

    private void menuEinstellungenActionPerformed(ActionEvent e)
    {
        // Klick auf den Menüpunkt "Einstellungen"
        Einstellungen config = new Einstellungen();
        config.setVisible(true);

        if (GuiManagerBean.getInstance().isStreamRunning())
        {
            log.warning("Änderungen an der Konfiguration sind erst für neue Streams aktiv!");
        }

        reloadConfig();
    }

    private void menuVerzeichnisseActionPerformed(ActionEvent e)
    {
        // Klick auf den Menüpunkt "Einstellungen"
        Verzeichnisse config = new Verzeichnisse(this);
        config.setVisible(true);

        if (GuiManagerBean.getInstance().isStreamRunning())
        {
            log.warning("Änderungen an der Konfiguration sind erst für neue Streams aktiv!");
        }
        reloadConfig();
    }

    private void menuQuitActionPerformed(ActionEvent e)
    {
        quitProgram();
    }

    private void quitProgram()
    {
        log.info("Anwendung wird geschlossen ...");
        if (chan != null)
        {
            this.chan.stopStreamBaseThread();
        }
        this.setVisible(false);
        this.dispose();
    }

    private void cbChannelItemStateChanged(ItemEvent e)
    {
        ChannelEntry ce = (ChannelEntry) e.getItem();
        tfSref.setText(channelIdList.getProperty(ce.getChannelPropertiesKey()));
        tfSref.setEditable(false);
        cbEpgType.setEnabled(false);
        
        switch (ce)
        {
            case FREI:
                tfSref.setText("");
                tfSref.setEditable(true);
                tfMusicDir.setText(configHandler.getConfig().getMusicFreieAuswahl());
                cbEpgType.setEnabled(true);
                break;
            case CE_60er_70er:
                cbEpgType.setSelectedItem(EPGType.SKY_RADIO);
                tfMusicDir.setText(configHandler.getConfig().getMusic60er70er());
                break;
            case CE_80er_90er:
                cbEpgType.setSelectedItem(EPGType.SKY_RADIO);
                tfMusicDir.setText(configHandler.getConfig().getMusic80er90er());
                break;
            case CE_Country:
                cbEpgType.setSelectedItem(EPGType.SKY_RADIO);
                tfMusicDir.setText(configHandler.getConfig().getMusicCountry());
                break;
            case CE_DeutscheCharts:
                cbEpgType.setSelectedItem(EPGType.SKY_RADIO);
                tfMusicDir.setText(configHandler.getConfig().getMusicDeutscheCharts());
                break;
            case CE_Lovesongs:
                cbEpgType.setSelectedItem(EPGType.SKY_RADIO);
                tfMusicDir.setText(configHandler.getConfig().getMusicLovesongs());
                break;
            case CE_RnB_Hiphop:
                cbEpgType.setSelectedItem(EPGType.SKY_RADIO);
                tfMusicDir.setText(configHandler.getConfig().getMusicRnbHiphop());
                break;
            case CE_RockHymnen:
                cbEpgType.setSelectedItem(EPGType.SKY_RADIO);
                tfMusicDir.setText(configHandler.getConfig().getMusicRockHymnen());
                break;
            default:
                cbEpgType.setSelectedItem(EPGType.SKY_RADIO);
                log.warning("Channeltyp unbekannt.");
                break;
        }
    }

    private void btnStreamActionPerformed(ActionEvent e)
    {
        if (validateStreamData())
        {
            modusAufnahme();
            startStreaming();
        }
    }

    private void btnAufnahmeverzActionPerformed(ActionEvent e)
    {
        JFileChooser fc = new JFileChooser(new File(tfMusicDir.getText()));
        fc.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        int returnVal = fc.showOpenDialog(this);
        if (returnVal == JFileChooser.APPROVE_OPTION)
        {
            File file = fc.getSelectedFile();
            tfMusicDir.setText(file.getAbsolutePath());
        }
    }

    private void btnStoppActionPerformed(ActionEvent e)
    {
        log.info("Stoppe Streaming-Thread aufgrund Anwenderwunsch ...");
        chan.stopStreamBaseThread();
        modusKeineAufnahme();
    }

    private void thisWindowClosing(WindowEvent e)
    {
        quitProgram();
    }

    private void initComponents()
    {
        // GEN-BEGIN:initComponents
        menuBar = new JMenuBar();
        menu = new JMenu();
        menuEinstellungen = new JMenuItem();
        menuVerzeichnisse = new JMenuItem();
        menuQuit = new JMenuItem();
        panelLog = new JPanel();
        scrollPane1 = new JScrollPane();
        logArea = new JTextArea();
        panelSelection = new JPanel();
        label2 = new JLabel();
        cbChannel = new JComboBox();
        label3 = new JLabel();
        tfSref = new JTextField();
        btnStream = new JButton();
        lblMusicDir = new JLabel();
        tfMusicDir = new JTextField();
        btnAufnahmeverz = new JButton();
        lblAufnahmeLaeuft = new JLabel();
        btnStopp = new JButton();
        lblEPGType = new JLabel();
        cbEpgType = new JComboBox();

        //======== this ========
        setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
        setIconImage(new ImageIcon(getClass().getResource("/icon_32.gif")).getImage());
        addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent e) {
                thisWindowClosing(e);
            }
        });
        Container contentPane = getContentPane();
        contentPane.setLayout(new BorderLayout());

        //======== menuBar ========
        {

            //======== menu ========
            {

                //---- menuEinstellungen ----
                menuEinstellungen.addActionListener(new ActionListener() {
                    @Override
                    public void actionPerformed(ActionEvent e) {
                        menuEinstellungenActionPerformed(e);
                    }
                });
                menu.add(menuEinstellungen);

                //---- menuVerzeichnisse ----
                menuVerzeichnisse.addActionListener(new ActionListener() {
                    @Override
                    public void actionPerformed(ActionEvent e) {
                        menuVerzeichnisseActionPerformed(e);
                    }
                });
                menu.add(menuVerzeichnisse);

                //---- menuQuit ----
                menuQuit.addActionListener(new ActionListener() {
                    @Override
                    public void actionPerformed(ActionEvent e) {
                        menuQuitActionPerformed(e);
                    }
                });
                menu.add(menuQuit);
            }
            menuBar.add(menu);
        }
        setJMenuBar(menuBar);

        //======== panelLog ========
        {
            panelLog.setLayout(new BorderLayout());

            //======== scrollPane1 ========
            {
                scrollPane1.setViewportBorder(null);

                //---- logArea ----
                logArea.setRows(25);
                logArea.setWrapStyleWord(true);
                logArea.setLineWrap(true);
                logArea.setEditable(false);
                logArea.setFont(new Font("Courier", Font.PLAIN, 12));
                logArea.setColumns(80);
                scrollPane1.setViewportView(logArea);
            }
            panelLog.add(scrollPane1, BorderLayout.CENTER);
        }
        contentPane.add(panelLog, BorderLayout.CENTER);

        //======== panelSelection ========
        {

            //---- cbChannel ----
            cbChannel.addItemListener(new ItemListener() {
                @Override
                public void itemStateChanged(ItemEvent e) {
                    cbChannelItemStateChanged(e);
                }
            });

            //---- btnStream ----
            btnStream.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    btnStreamActionPerformed(e);
                }
            });

            //---- btnAufnahmeverz ----
            btnAufnahmeverz.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    btnAufnahmeverzActionPerformed(e);
                }
            });

            //---- lblAufnahmeLaeuft ----
            lblAufnahmeLaeuft.setForeground(Color.red);

            //---- btnStopp ----
            btnStopp.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    btnStoppActionPerformed(e);
                }
            });

            //---- cbEpgType ----
            cbEpgType.setEnabled(false);

            GroupLayout panelSelectionLayout = new GroupLayout(panelSelection);
            panelSelection.setLayout(panelSelectionLayout);
            panelSelectionLayout.setHorizontalGroup(
                panelSelectionLayout.createParallelGroup()
                    .addGroup(panelSelectionLayout.createSequentialGroup()
                        .addGap(19, 19, 19)
                        .addGroup(panelSelectionLayout.createParallelGroup(GroupLayout.Alignment.LEADING, false)
                            .addGroup(panelSelectionLayout.createSequentialGroup()
                                .addGroup(panelSelectionLayout.createParallelGroup()
                                    .addComponent(lblMusicDir)
                                    .addComponent(lblAufnahmeLaeuft))
                                .addPreferredGap(LayoutStyle.ComponentPlacement.RELATED)
                                .addGroup(panelSelectionLayout.createParallelGroup()
                                    .addGroup(panelSelectionLayout.createSequentialGroup()
                                        .addComponent(btnStream)
                                        .addPreferredGap(LayoutStyle.ComponentPlacement.UNRELATED)
                                        .addComponent(btnStopp))
                                    .addGroup(panelSelectionLayout.createSequentialGroup()
                                        .addComponent(tfMusicDir, GroupLayout.PREFERRED_SIZE, 550, GroupLayout.PREFERRED_SIZE)
                                        .addGap(18, 18, 18)
                                        .addComponent(btnAufnahmeverz))))
                            .addGroup(panelSelectionLayout.createSequentialGroup()
                                .addGroup(panelSelectionLayout.createParallelGroup()
                                    .addComponent(label2, GroupLayout.PREFERRED_SIZE, 146, GroupLayout.PREFERRED_SIZE)
                                    .addComponent(label3))
                                .addPreferredGap(LayoutStyle.ComponentPlacement.RELATED)
                                .addGroup(panelSelectionLayout.createParallelGroup()
                                    .addComponent(tfSref, GroupLayout.PREFERRED_SIZE, 550, GroupLayout.PREFERRED_SIZE)
                                    .addGroup(panelSelectionLayout.createSequentialGroup()
                                        .addComponent(cbChannel, GroupLayout.PREFERRED_SIZE, 199, GroupLayout.PREFERRED_SIZE)
                                        .addGap(36, 36, 36)
                                        .addComponent(lblEPGType, GroupLayout.PREFERRED_SIZE, 104, GroupLayout.PREFERRED_SIZE)
                                        .addPreferredGap(LayoutStyle.ComponentPlacement.RELATED)
                                        .addComponent(cbEpgType, GroupLayout.PREFERRED_SIZE, 199, GroupLayout.PREFERRED_SIZE)))))
                        .addGap(100, 100, 100))
            );
            panelSelectionLayout.linkSize(SwingConstants.HORIZONTAL, new Component[] {label2, lblMusicDir});
            panelSelectionLayout.setVerticalGroup(
                panelSelectionLayout.createParallelGroup()
                    .addGroup(panelSelectionLayout.createSequentialGroup()
                        .addGap(18, 18, 18)
                        .addGroup(panelSelectionLayout.createParallelGroup()
                            .addGroup(panelSelectionLayout.createParallelGroup(GroupLayout.Alignment.BASELINE)
                                .addComponent(label2)
                                .addComponent(cbChannel, GroupLayout.PREFERRED_SIZE, GroupLayout.DEFAULT_SIZE, GroupLayout.PREFERRED_SIZE))
                            .addGroup(panelSelectionLayout.createParallelGroup(GroupLayout.Alignment.BASELINE)
                                .addComponent(lblEPGType)
                                .addComponent(cbEpgType, GroupLayout.PREFERRED_SIZE, GroupLayout.DEFAULT_SIZE, GroupLayout.PREFERRED_SIZE)))
                        .addPreferredGap(LayoutStyle.ComponentPlacement.RELATED)
                        .addGroup(panelSelectionLayout.createParallelGroup(GroupLayout.Alignment.BASELINE)
                            .addComponent(label3)
                            .addComponent(tfSref, GroupLayout.PREFERRED_SIZE, GroupLayout.DEFAULT_SIZE, GroupLayout.PREFERRED_SIZE))
                        .addPreferredGap(LayoutStyle.ComponentPlacement.RELATED)
                        .addGroup(panelSelectionLayout.createParallelGroup(GroupLayout.Alignment.BASELINE)
                            .addComponent(lblMusicDir)
                            .addComponent(tfMusicDir, GroupLayout.PREFERRED_SIZE, GroupLayout.DEFAULT_SIZE, GroupLayout.PREFERRED_SIZE)
                            .addComponent(btnAufnahmeverz))
                        .addPreferredGap(LayoutStyle.ComponentPlacement.UNRELATED)
                        .addGroup(panelSelectionLayout.createParallelGroup(GroupLayout.Alignment.BASELINE)
                            .addComponent(lblAufnahmeLaeuft)
                            .addComponent(btnStream)
                            .addComponent(btnStopp))
                        .addContainerGap(GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
            );
        }
        contentPane.add(panelSelection, BorderLayout.NORTH);
        setSize(900, 600);
        setLocationRelativeTo(null);

        initComponentsI18n();
        // GEN-END:initComponents
        
//        panelSelection.setPreferredSize(new Dimension(400, 200));
    }

    private void initComponentsI18n()
    {
        // GEN-BEGIN:initI18n
        ResourceBundle bundle = ResourceBundle.getBundle("ui_dialog");
        setTitle(bundle.getString("MainWindow.this.title"));
        menu.setText(bundle.getString("MainWindow.menu.text"));
        menuEinstellungen.setText(bundle.getString("MainWindow.menuEinstellungen.text"));
        menuVerzeichnisse.setText(bundle.getString("MainWindow.menuVerzeichnisse.text"));
        menuQuit.setText(bundle.getString("MainWindow.menuQuit.text"));
        label2.setText(bundle.getString("MainWindow.label2.text"));
        cbChannel.setToolTipText(bundle.getString("MainWindow.cbChannel.toolTipText"));
        label3.setText(bundle.getString("MainWindow.label3.text"));
        label3.setToolTipText(bundle.getString("MainWindow.label3.toolTipText"));
        tfSref.setToolTipText(bundle.getString("MainWindow.tfSref.toolTipText"));
        btnStream.setText(bundle.getString("MainWindow.btnStream.text"));
        btnStream.setToolTipText(bundle.getString("MainWindow.btnStream.toolTipText"));
        lblMusicDir.setText(bundle.getString("MainWindow.lblMusicDir.text"));
        tfMusicDir.setToolTipText(bundle.getString("MainWindow.tfMusicDir.toolTipText"));
        btnAufnahmeverz.setText(bundle.getString("MainWindow.btnAufnahmeverz.text"));
        lblAufnahmeLaeuft.setText(bundle.getString("MainWindow.lblAufnahmeLaeuft.text"));
        btnStopp.setText(bundle.getString("MainWindow.btnStopp.text"));
        btnStopp.setToolTipText(bundle.getString("MainWindow.btnStopp.toolTipText"));
        lblEPGType.setText(bundle.getString("MainWindow.lblEPGType.text"));
        cbEpgType.setToolTipText(bundle.getString("MainWindow.cbEpgType.toolTipText"));
        // GEN-END:initI18n
    }

    // GEN-BEGIN:variables
    private JMenuBar menuBar;
    private JMenu menu;
    private JMenuItem menuEinstellungen;
    private JMenuItem menuVerzeichnisse;
    private JMenuItem menuQuit;
    private JPanel panelLog;
    private JScrollPane scrollPane1;
    private JTextArea logArea;
    private JPanel panelSelection;
    private JLabel label2;
    private JComboBox cbChannel;
    private JLabel label3;
    private JTextField tfSref;
    private JButton btnStream;
    private JLabel lblMusicDir;
    private JTextField tfMusicDir;
    private JButton btnAufnahmeverz;
    private JLabel lblAufnahmeLaeuft;
    private JButton btnStopp;
    private JLabel lblEPGType;
    private JComboBox cbEpgType;
    // GEN-END:variables

    @Override
    public void publishLogRecord(String logRecord)
    {
        // Maximale Zeichenzahl in der GUI wegen Speicherverbrauch begrenzen ...
        StringBuilder sb = new StringBuilder();
        sb.append(logArea.getText());
        sb.append(logRecord);
        int diff = sb.length() - MAX_CHAR_LOG_BUFFER;
        if (diff > 0)
        {
            sb.delete(0, diff);
        }
        logArea.setText(sb.toString());

        // An das Ende des Textes scrollen
        logArea.scrollRectToVisible(new Rectangle(0, logArea.getHeight() - 2, 1, 1));

    }

    @Override
    public void exceptionOccured(Exception e)
    {
        log.severe("Stoppe Streaming : " + e.getMessage());
        modusKeineAufnahme();
    }
}
