package mpeg2demux;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.StringWriter;

/**
 * User: Andreas Rosenberg
 * Date: 14.09.2003
 * Time: 15:54:09
 * This class represents a packetized elementary stream packet.
 * Usually the data contained as payload is an embedded MPEG2 stream.
 * Changes: 2004.01.07
 *   - PES video packet headers may contain padding bytes
 *     skip padding bytes
 */
public class MPEG2PESPacket extends MPEG2Packet implements MPEG2PacketInterface{

    public boolean isMetaStreamPacket(){
        return (0xE0&type) == 0xE0 || (0xE0&type) == 0xC0;
    }
    public boolean isVideoPacket(){
        return (0xE0&type) == 0xE0;
    }
    public boolean isAudioPacket(){
        return (0xE0&type) == 0xC0;
    }
    public boolean hasPTS(){
        return data != null && size>0 && (128&data[1])>0;
    }
    public boolean hasDTS(){
        return data != null && size>0 && (64&data[1])>0;
    }
    public long getPTS(){
        if ((128&data[1]) >0){
            return 0xFFFFFFFFL & (14L&data[3])<<28 | (255L&data[4])<<22 | (254L&data[5])<<14
                  | (255L&data[6])<<7 | (254&data[7]) >>> 1;
        }
        return 0;
        }
    public long getDTS(){
        if ((128&data[1]) >0){
            return 0xFFFFFFFFL & (14L&data[8])<<28 | (255L&data[9])<<22 | (254L&data[10])<<14
                  | (255L&data[11])<<7 | (254&data[12]) >>> 1;
        }
        return 0;
        }

    public void readPacket(InputStream in,MPEG2Parser parser) throws java.io.IOException {
        byte tmp[] = new byte[2];
        int psize;

        in.read(tmp,0,2);
        psize = ((255&tmp[0])<<8) + (255&tmp[1]);
        size = psize;
        data = new byte[size];
        in.read(data,0,size);
        System.out.println(info());
    }

    public String name() {
        String name;
        name = "MPEG2PESPacket";
        if ((type&0xE0) == 0xE0 )
           name = name + "(V)" + Integer.toHexString(255&type);
        else if ((type&0xC0) == 0xC0)
            name = name + "(A)" + Integer.toHexString(255&type);
        if (size > 0 && (64&data[1]) >0) name = name + " DTS";
        return name;
    }
    public String info() {
        int i,flags;
        StringWriter writer = new StringWriter();

        writer.write("MPEG2PESPacket:");
        if ((type&0xE0) == 0xE0 )
           writer.write("Video:");
        else if ((type&0xC0) == 0xC0)
            writer.write("Audio:");
        writer.write(" Size:"+Long.toHexString(size)+"\n");
        writer.write("Flags(1):"+Integer.toBinaryString(255&data[0])+"\n");
        flags = 255&data[1];
        writer.write("Flags(2):"+Integer.toBinaryString(flags));
        if ((128&flags) >0) writer.write(" PTS");
        if ((64&flags) >0) writer.write(" DTS");
        if ((32&flags) >0) writer.write(" ESCR");
        if ((16&flags) >0) writer.write(" ES-rate");
        if ((8&flags) >0) writer.write(" DSM trick mode");
        if ((4&flags) >0) writer.write(" add (c) info");
        if ((2&flags) >0) writer.write(" CRC");
        if ((1&flags) >0) writer.write(" Extension");
        writer.write("\nPES Header length"+Integer.toHexString(255&data[4])+"\n");
        i = 3;
        if ((128&flags) >0){
            long PTS;
            PTS = 0xFFFFFFFFL & (14L&data[i])<<28 | (255L&data[i+1])<<22 | (254L&data[i+2])<<14
                  | (255L&data[i+3])<<7 | (254&data[i+4]) >>> 1;
            int PTSf = (240&data[i])>>>4;
            writer.write("PTS: "+Long.toHexString(PTS)+" "+MpegTimeStampToString(PTS)+" PTS-Flags: "+Integer.toBinaryString(PTSf));
            i += 5;
        }
        if ((64&flags) >0){
            long DTS;
            DTS = 0xFFFFFFFFL & (14L&data[i])<<28 | (255L&data[i+1])<<22 | (254L&data[i+2])<<14
                  | (255L&data[i+3])<<7 | (254&data[i+4]) >>> 1;
            int DTSf = (240&data[i])>>>4;
            writer.write(" DTS: "+Long.toHexString(DTS)+" "+MpegTimeStampToString(DTS)+" DTS-Flags: "+Integer.toBinaryString(DTSf));
            i += 5;
        }
        if ((32&flags) >0){
            long ESCR;
            ESCR = 0xFFFFFFFFL & (66L&data[i])<<26 | (3L&data[i])<<28 | (255L&data[i+1])<<20
                   | (248L&data[i+2])<<12 | (3L&data[i+2])<<13 | (255L&data[i+3])<<5
                   | (248L&data[i+4])>>3;
            writer.write(" ESCR: "+Long.toHexString(ESCR));
            i += 6;
        }
        writer.write("\n");
        if ((16&flags) >0){
            long ESR;
            ESR = 0xFFFFFFFFL & (127&data[i])<<15 | (255&data[i+1])<<7 | (254&data[i+2])>>>1;
            writer.write(" ES Rate: "+Long.toHexString(ESR*50)+"b/s");
            i += 6;
        }
        if ((8&flags) >0){
            i += 4;
        }
        if ((4&flags) >0){
            i += 1;
        }
        if ((2&flags) >0){
            i += 2;
        }
        if ((1&flags) >0){
            flags = 255&data[i];
            i += 1;
            writer.write("\nFlags(3):"+Integer.toBinaryString(flags)+" ");
            if ((128&flags) >0) writer.write(" Priv.Data");
            if ((64&flags) >0) writer.write(" Pack Header");
            if ((32&flags) >0) writer.write(" Seq.Counter");
            if ((16&flags) >0) writer.write(" P-STD Buffer");
            if ((1&flags) >0) writer.write(" PES-Extension");
            writer.write("\n");

            if ((128&flags) >0) {
                i += 128;
            }
            if ((64&flags) >0) {
                int l = 255&data[i];
                writer.write("Pack length:"+Integer.toHexString(l));
                i += l + 1;
            }
            if ((32&flags) >0) {
                int l = 127&data[i];
                writer.write("Seq.Counter:"+Integer.toHexString(l));
            l = 127&data[i+1];
            writer.write("Stuffing length:"+Integer.toHexString(l));
            i += 2;
            }
            if ((16&flags) >0) {
                int l,m=128;
                l = 255&data[i];
                if ((l&32)>0) {
                    writer.write("Scaled");
                    m = 1024;
                }
                l = ((l&31)<<8 | (255&data[i]))*m;
                writer.write("P-STD Buff-Size:"+Integer.toHexString(l));
                i += 2;
            }
            if ((1&flags) >0) {
                int l = 255&data[i];
                writer.write("Ext.Pack length:"+Integer.toHexString(l));
                i += l + 1;
            }
        }
        writer.write("\nSize of headers:"+Integer.toHexString(i));
        return writer.toString();
    }
    /* returns the size of header information exluding the 0x000001xx packet header */
    public int sizePacketHeader() {
        int i,pad;
        if (data == null || size == 0)
            return 0;

        i = data[2]+3;
        if ((type&0xE0) == 0xE0 && (data[i] != 0)) {
            pad = this.skipStuffingBytes(data,i,size);
            if (pad > 0)
               i += pad;
        }
        return i;
    }

    public int skipStuffingBytes(byte [] in, int start, int stop) {
        int i=start;

        while (i < stop) {
            if (in[i] == -1)
               i++;
            else
               break;
        }
        return (i-start);
    }

    public boolean parseEncapsulatedData(MPEG2PacketStreamListener listener, MPEG2Addable collector){
        boolean res=true;
        MPEG2Parser parser = new MPEG2Parser();
        BufferedInputStream in = new BufferedInputStream(new ByteArrayInputStream(this.getData()));
        try {
            res = parser.parsePStream(in,listener, collector);
        } catch (java.io.IOException e) {
            System.out.println("Exception:" + e);
        }
        return res;
    }
}
