package mpeg2demux;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;

/**
 * User: Andreas Rosenberg
 * Date: 14.09.2003
 * Time: 16:03:09
 * A simple MPEG2 parser, that logs detected packets on System.out.
 */
public class MPEG2Parser implements MPEG2PacketStreamListener {
    public static final int MPEG2ProgramStreamSystemHeader=0xBB;
    public static final int MPEG2ProgramPack=0xBA;
    public static final int MPEG2ProgramEndCode=0xB9;
    public static final int MPEG2GroupHeader=0xB8;
    public static final int MPEG2SequenceEndCode=0xB7;
    public static final int MPEG2ExtensionHeader=0xB5;
    public static final int MPEG2SequenceHeader=0xB3;
    public static final int MPEG2UserData=0xB2;
    public static final int MPEG2PictureHeader=0x00;

    public static final int MPEG2ESequenceExtensionID=0x01;
    public static final int MPEG2ESequenceDisplayExtensionID=0x02;
    public static final int MPEG2EQuantMatrixExtensionID=0x03;
    public static final int MPEG2ECopyrightExtensionID=0x04;
    public static final int MPEG2ESequenceScalableExtensionID=0x05;
    public static final int MPEG2EPictureDisplayExtensionID=0x07;
    public static final int MPEG2EPictureCodingExtensionID=0x08;
    public static final int MPEG2EPictureSpatialScalableExtensionID=0x09;
    public static final int MPEG2EPictureTemporalScalableExtensionID=0x0A;
    
    private static final int scanSize = 4096;
    private byte[] scanbuffer = new byte[scanSize];

    public boolean onPacketReceive(MPEG2PacketInterface packet, long position, MPEG2Addable collector){
        System.out.println("Position: "+Long.toHexString(position)+" "+packet.name());
        return true;
    }

    public static void main(String[] args) {
        MPEG2Parser mpeg2parser = new MPEG2Parser();
        switch (args.length){
            case 1:
                mpeg2parser.parsePESFile(args[0],mpeg2parser,null);
                break;
            default:
                System.out.println("Usage: MPEG2Parser file\n");
        }
    }

    void parsePESFile(String filename, MPEG2PacketStreamListener listener,MPEG2Addable collector){
        File file = new File(filename);

        try {
        if (file.exists()){
            FileInputStream fileInputStream = new FileInputStream(file);
            BufferedInputStream in = new BufferedInputStream(fileInputStream,scanSize);
            this.parsePEStream(in,listener, collector);
            in.close();
        }
        } catch (java.io.IOException e) {
            System.out.println("Exception:"+e);
        }
    }
    /* scan for the sequence 0x000001 in the input stream, by scanning max bytes:
    returns the number of bytes skipped (excluding the found sequence)
    returns max negated if the sequence could not be found in range defined by max */

    public long scanForMPEGHeader(InputStream in, int max) throws java.io.IOException {
        long counter=0;
        int index=0, b;
        byte queue[] = new byte[3] ;

        while ((b = in.read()) >= 0 && counter < max) {
            queue[index] = (byte) b;
            index++;
            counter++;
            if (index == 3){
                if (queue[0] == 0 && queue[1]==0 && queue[2] == 1)
                {
                    return (int)counter-3;
                }
                queue[0] = queue[1];
                queue[1] = queue[2];
                index = 2;
            }
        }
        return -counter;
    }

    /* reads bytes into the data array, until a start code sequence appears
    Returns the number of bytes read - excluding the start code sequence.*/
    public long readUptoMPEGHeader(InputStream in, int max, byte[] data) throws java.io.IOException {
        long counter=0;
        int index=0, b;

        while ((b = in.read()) >= 0 && counter < max) {
            data[index] = (byte) b;
            counter++;
            if (index > 2){
                if (data[index-2] == 0 && data[index-1]==0 && data[index] == 1)
                    return (int)counter-3;
            }
            index++;
        }
        return -counter;
    }

    /* Skip bytes until a new start code apprears in the stream.
    Returns the type of the start code.*/

    public long scanForNextStartCode(InputStream in, int max) throws java.io.IOException {
        long startcode;

        startcode = scanForMPEGHeader(in, max);
        if (startcode >= 0) {
            return in.read();
        }
        return startcode;
    }

    /* This is the generic parsing routine:
    - in is the stream to be parsed
    - scanSize defines the size being scanned for a new packet header
    - listener is being notified with #onPacketReceive if a new packet has been decoded
    - addable is being passed together with other args to #onPacketReceive - so you can collect the packets somewhere

    the boolean returned is the return value of the listener -
    If the listener want's to stop the parse process it should return false
    If the functions returns true, the input file has been read up to EOF
    */
    public boolean parsePEStream(InputStream in, MPEG2PacketStreamListener listener, MPEG2Addable collector) throws java.io.IOException {
        long counter = 0,offset=0;
        int ptype=-1;
        boolean cont=true,scanForNextStartCode=true;
        MPEG2PacketInterface p;

        do {
            if (scanForNextStartCode) {
                offset = scanForNextStartCode(in, scanSize);
                if (offset >= 0) {
                    ptype = (int)offset;
                    if ((ptype & 0xE0) == 0xE0 || (ptype & 0xE0) == 0xC0) {
                        p = new MPEG2PESPacket();
                        p.setFilePos(counter);
                        p.setType(ptype);
                        p.readPacket(in, this);
                        scanForNextStartCode = true;
                        if (listener != null)
                            cont=listener.onPacketReceive(p, counter, collector);
                        counter += p.size();
                    } else {
                        System.out.print("1> Undecoded packet:" + Integer.toHexString(ptype));
                    }
                }
            }
            if (!cont)
                return cont;
        }
        while (offset >= 0 || (offset == -scanSize));
        return true;
    }

    /**
     * @param in
     * @param listener
     * @param collector
     * @return
     * @throws java.io.IOException
     */
    public boolean parsePStream(InputStream in, MPEG2PacketStreamListener listener, MPEG2Addable collector) throws java.io.IOException {
        long counter = 0,offset=0;
        int ptype=-1,petype=-1,size;
        boolean cont=true,scanForNextStartCode=true;
        byte[] tmp;
        MPEG2PacketInterface p = null;

            
        do {
            if (scanForNextStartCode) {
                offset = scanForNextStartCode(in, scanSize);
                if (offset >= 0) {
                    ptype = (int)offset;
                    }
                else
                    ptype = -1;
                }
            switch (ptype) {
                case -1:
                    break;
                case MPEG2ProgramPack:
                    p = new MPEG2ProgramPack();
                    p.setFilePos(counter);
                    p.readPacket(in, this);
                    scanForNextStartCode = true;
                    if (listener != null)
                        cont=listener.onPacketReceive(p, counter, collector);
                    counter += p.size();
                    break;
                case MPEG2ProgramEndCode:
                    p = new MPEG2ProgramEndCode();
                    p.setFilePos(counter);
                    p.readPacket(in, this);
                    scanForNextStartCode = true;
                    if (listener != null)
                        cont=listener.onPacketReceive(p, counter, collector);
                    counter += p.size();
                    break;
                case MPEG2GroupHeader:
                    p = new MPEG2GroupHeader();
                    p.setFilePos(counter);
                    p.readPacket(in, this);
                    scanForNextStartCode = true;
                    if (listener != null)
                        cont=listener.onPacketReceive(p, counter, collector);
                    counter += p.size();
                    break;
                case MPEG2ProgramStreamSystemHeader:
                    p = new MPEG2ProgramStreamSystemHeader();
                    p.setFilePos(counter);
                    p.readPacket(in, this);
                    scanForNextStartCode = true;
                    if (listener != null)
                        cont=listener.onPacketReceive(p, counter, collector);
                    counter += p.size();
                    break;
                case MPEG2SequenceEndCode:
                    p = new MPEG2SequenceEndCode();
                    p.setFilePos(counter);
                    p.readPacket(in, this);
                    scanForNextStartCode = true;
                    if (listener != null)
                        cont=listener.onPacketReceive(p, counter, collector);
                    counter += p.size();
                    break;
                case MPEG2SequenceHeader:
                    p = new MPEG2SequenceHeader();
                    p.setFilePos(counter);
                    p.readPacket(in, this);
                    if (listener != null)
                        cont=listener.onPacketReceive(p, counter, collector);
                    counter += p.size();
                    /* a sequence extension must follow a sequence header -
                    otherwise it's MPEG 1 (?) */
                    offset = scanForNextStartCode(in, scanSize);
                    if (offset >= 0) {
                        petype = (int)offset;
                    }
                    while (petype == MPEG2ExtensionHeader && !cont) {
                        p = new MPEG2Extension();
                        p.setFilePos(counter);
                        p.setType((byte) petype);
                        size = (int)readUptoMPEGHeader(in, scanSize,scanbuffer);
                        if (size >= 0) {
                            tmp = new byte[size];
                            System.arraycopy(scanbuffer,0,tmp,0,size);
                            p.setData(tmp);
                            p.setSize(size);
                            if (listener != null)
                                cont= cont && listener.onPacketReceive(p, counter, collector);
                            counter += size;
                            petype = in.read();
                        }else {
                            System.out.print("Sequence extension header with wrong size:" + Long.toHexString(counter));
                        }
                    }
                    scanForNextStartCode = false;
                    ptype = petype;
                    break;
                case MPEG2PictureHeader:
                    p = new MPEG2PictureHeader();
                    p.setFilePos(counter);
                    p.readPacket(in, this);
                    if (listener != null)
                        cont=listener.onPacketReceive(p, counter, collector);
                    counter += p.size();
                    offset = scanForNextStartCode(in, scanSize);
                    if (offset >= 0) {
                        petype = (int)offset;
                    }
                    while (petype == MPEG2ExtensionHeader && !cont) {
                        p = new MPEG2Extension();
                        p.setType((byte) petype);
                        p.setFilePos(counter);
                        size = (int)readUptoMPEGHeader(in, scanSize,scanbuffer);
                        if (size >= 0) {
                            tmp = new byte[size];
                            System.arraycopy(scanbuffer,0,tmp,0,size);
                            p.setData(tmp);
                            p.setSize(size);
                            if (listener != null)
                                cont=cont && listener.onPacketReceive(p, counter, collector);
                            counter += size;
                            petype = in.read();
                        } else {
                            System.out.print("Picture Extension header with wrong size:" + Long.toHexString(counter));
                        }
                    }
                    scanForNextStartCode = false;
                    ptype = petype;
                    break;

                case MPEG2ExtensionHeader:
                    p = new MPEG2Extension();
                    p.setType((byte) ptype);
                    p.setFilePos(counter);
                    size = (int)readUptoMPEGHeader(in, scanSize,scanbuffer);
                    if (size >= 0) {
                        tmp = new byte[size];
                        System.arraycopy(scanbuffer,0,tmp,0,size);
                        p.setData(tmp);
                        p.setSize(size);
                        if (listener != null)
                            cont=cont && listener.onPacketReceive(p, counter, collector);
                        counter += size;
                        petype = in.read();
                    } else {
                        System.out.print("Extension header with wrong size:" + Long.toHexString(counter));
                    }
                    scanForNextStartCode = false;
                    ptype = petype;
                    break;

                default:
                    if ((ptype & 0xE0) == 0xE0 || (ptype & 0xE0) == 0xC0) {
                        System.out.print("PES packet in program stream:" + Long.toHexString(counter));
                    } else if (ptype > 0 && ptype < 0xB0) {
                        p = new MPEG2PictureSlice();
                        p.setFilePos(counter);
                        p.setType((byte) ptype);
                        size = (int)readUptoMPEGHeader(in, scanSize,scanbuffer);
                        if (size >= 0) {
                            tmp = new byte[size];
                            System.arraycopy(scanbuffer,0,tmp,0,size);
                            p.setData(tmp);
                            p.setSize(size);
                            if (listener != null)
                                cont = listener.onPacketReceive(p, counter, collector);
                            counter += size;
                            ptype = in.read();
                            scanForNextStartCode = false;
                        } else {
                            /* System.out.print("Picture Slice with wrong size:" + Long.toHexString(counter) + " "+ Integer.toHexString(size)); */
                            scanForNextStartCode = true;
                        }
                    } else {
                        System.out.print("2> Undecoded packet:" + Integer.toHexString(ptype));
                    }
            }
            if (!cont)
                return cont;
        }
        while (offset >= 0 || (offset == -scanSize));
        return true;
    }
}
