package mpeg2demux;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;

/**
 * User: Andreas Rosenberg  - This is my first "real" Java program.
 * Date: 19.09.2003
 * Time: 15:48:06
 * This is a simple Demuxer for MPEG2 PES streams.
 *
 * This program is free software; you can redistribute it free of charge
 * and/or modify it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program comes without any warranty.
 * It is primarily intended for educational purposes.
 *
 * Theory of operation:
 *   A PES stream is sequence of PES packets. Each PES packet has a specific header
 *   and a "payload". The payload is another MPEG2 stream (i.e. a program stream).
 *   If the payload is video or audio data is determined by the packet id. (look
 *   at MPEG2PESPacket isVideoPacket, isAudioPacket). The demuxer only stores the
 *   different kind of payloads on a specific stream. It is important to start storing
 *   of each stream only with packets that have a PTS flag. Otherwise the result may be
 *   unusable, because uncomplete packets of the embedded stream may exist. The delay
 *   between the first written video and audio packet is stored and printed at the end
 *   of the process. This allows you to use this delay for a muxer.
 *
 * Changes: 2004.01.07
 *   - first audio packet should begin with a frame header 0xFFFC
 *     skip data upto frame header
 *
 * Changes: 2004.03.01
 *   - previous versions shortened a lot of picture slices, which results
 *     in a mostly corrupted video file. Several changes throughout the
 *     whole project cleaned things up a bit.
 */
public class MPEG2StreamDemuxer implements MPEG2PacketStreamListener {
    static private int BufferSize=4096;
    protected BufferedOutputStream audioOut;
    protected BufferedOutputStream videoOut;
    protected ArrayList infiles=new ArrayList();;
    static private boolean audioPTS=false,videoPTS=false;
    static private long firstAudioPTS,firstVideoPTS;
    private int packetCounter=0;

    public BufferedOutputStream getAudioOut() {
        return audioOut;
    }

    public void setAudioOut(BufferedOutputStream out) {
        audioOut = out;
    }

    public BufferedOutputStream getVideoOut() {
        return videoOut;
    }

    public void setVideoOut(BufferedOutputStream out) {
        videoOut = out;
    }
    private void closeOutputStreams() throws java.io.IOException {
        if (videoOut != null)
            videoOut.close();
        if (audioOut != null)
            audioOut.close();
    }
    private long videoAudioDelay(){
        long msec = (firstVideoPTS-firstAudioPTS)/90;
        return msec;
    }

    public int scanForMPEGAudioHeaderArray(byte [] in, int start, int stop) {
        int index=0,i=start;
        byte queue[] = new byte[3];

        while (i < stop) {
            queue[index] = in[i];
            index++;
            i++;
            if (index == 2){
                if (queue[0] == -1 && queue[1]== -4)
                    return i-start-2;
                queue[0] = queue[1];
                index = 1;
            }
        }
        return -(i-start);
    }

    static public boolean determineAVDelay(MPEG2PacketInterface packet) {
        if (packet.isAudioPacket() ) {
            if (packet.hasPTS()) {
                if (!audioPTS)
                    firstAudioPTS = packet.getPTS();
                audioPTS = true;
            }
        }
        if (packet.isVideoPacket()) {
            if (packet.hasPTS()) {
                if (!videoPTS)
                    firstVideoPTS = packet.getPTS();
                videoPTS = true;
            }
        }
        return (videoPTS && audioPTS);
    }

    public boolean onPacketReceive(MPEG2PacketInterface packet, long position, MPEG2Addable collector) {
        int sizeHeader,sizeData,skip;
        sizeHeader = packet.sizePacketHeader();
        sizeData = packet.size() - sizeHeader;
        packetCounter++;
        try {
            if (packet.isMetaStreamPacket()) {
                if (packet.isAudioPacket() && audioOut != null) {
                    if (packet.hasPTS() || audioPTS) {
                        if (!audioPTS) {
                            firstAudioPTS = packet.getPTS();
                            skip = scanForMPEGAudioHeaderArray(packet.getData(),sizeHeader,packet.size());
                            if (skip > 0) {
                               sizeHeader += skip;
                               sizeData -= skip;
                            }
                        }
                        audioOut.write(packet.getData(), sizeHeader, sizeData);
                        audioPTS = true;
                    }
                }
                if (packet.isVideoPacket() && videoOut != null) {
                    if (packet.hasPTS() || videoPTS) {
                        videoOut.write(packet.getData(), sizeHeader, sizeData);
                        if (!videoPTS)
                            firstVideoPTS = packet.getPTS();
                        videoPTS = true;
                    }
                }
            } else {
                System.out.println("This stream must be a PES stream. Demuxing aborted");
                return false;
            }
        } catch (java.io.IOException e) {
            System.out.println("Exception:" + e);
        }
        return true;
    }

    private void demuxFile(String inFilename, MPEG2PacketStreamListener listener,MPEG2Addable collector){
        MPEG2Parser parser = new MPEG2Parser();
        parser.parsePESFile(inFilename,listener,collector);
    }

    private void printUsage(){
        System.out.println("Usage: MPEG2StreamDemuxer -i\"file1[,file2]\" -v\"videoout\" -a\"audioout\"\n");
    }

    private boolean parseArg(String argString, char actionChar,ArrayList filenames) {
        StringReader reader = new StringReader(argString);
        StringWriter writer = new StringWriter();
        int state = 0;
        int c,ac;

        try {
            while (reader.ready()) {
                switch (state) {
                    case 0:
                    c = (char) reader.read();
                    if (c>=0 && c == '-') {
                        ac = (char) reader.read();
                        if (ac == actionChar) {
                            state = 1;
                        }
                        else return false;
                    }
                    break;
                    default:
                        c = reader.read();
                        if (c>=0){
                            if (c == ','){
                                filenames.add(writer.toString());
                                writer = new StringWriter();
                            } else {
                                writer.write(c);
                            }
                        } else {
                            filenames.add(writer.toString());
                            return true;
                        }
                        break;
                }
            }
        } catch (java.io.IOException e) {
            System.out.println("Exception:" + e);
        }
        return false;
    }

    public static void main(String[] args) {
        MPEG2StreamDemuxer demuxer = new MPEG2StreamDemuxer();
        ArrayList outfilesVideo = new ArrayList(),outfilesAudio = new ArrayList();
        File videoFile,audioFile;
        FileOutputStream videoOutputStream,audioOutputStream;
        BufferedOutputStream videoOut,audioOut;
        int i;

        System.out.println("MPEG2StreamDemuxer V1.2 - 2004-03-07 written by A.Rosenberg");
        System.out.println("       send bugs and comments to mpeg2demuxer@rosen-berg.de\n");
        try {
            switch (args.length) {
                case 1: /* show only info of infile */
                    if (demuxer.parseArg(args[0], 'i', demuxer.infiles)) {
                        MPEG2PacketStreamListener localListener;
                        localListener = new MPEG2PacketStreamListener() {
                            MPEG2PacketStreamListener localListener2 = new MPEG2PacketStreamListener() {
                                public boolean onPacketReceive(MPEG2PacketInterface packet, long position, MPEG2Addable collector) {
                                    if (packet.type() == MPEG2Parser.MPEG2SequenceHeader) {
                                        System.out.println(packet.info());
                                    }
                                    return true;
                                }
                            };

                            public boolean onPacketReceive(MPEG2PacketInterface packet, long position, MPEG2Addable collector) {
                                if (packet.isMetaStreamPacket())
                                    packet.parseEncapsulatedData(localListener2, null);
                                else {
                                    if (packet.type() == MPEG2Parser.MPEG2SequenceHeader) {
                                        System.out.println(packet.info());
                                    }
                                }
                            return !MPEG2StreamDemuxer.determineAVDelay(packet);
                            }
                        };
                        for (i = 0; i < demuxer.infiles.size(); i++) {
                            System.out.println(demuxer.infiles.get(i) + "\n");
                            demuxer.demuxFile((String) demuxer.infiles.get(i), localListener, null);
                            System.out.println("Video/Audio delay (ms):" + demuxer.videoAudioDelay());
                        }
                    } else
                        demuxer.printUsage();
                    break;
                case 2: /* only one part should be extracted */
                    if (demuxer.parseArg(args[0], 'i', demuxer.infiles)) {
                        if (demuxer.parseArg(args[1], 'v', outfilesVideo)) {
                            videoFile = new File((String) outfilesVideo.get(0));
                            videoOutputStream = new FileOutputStream(videoFile);
                            videoOut = new BufferedOutputStream(videoOutputStream, BufferSize);
                            demuxer.setVideoOut(videoOut);
                        } else {
                            if (demuxer.parseArg(args[1], 'a', outfilesAudio)) {
                                audioFile = new File((String) outfilesAudio.get(0));
                                audioOutputStream = new FileOutputStream(audioFile);
                                audioOut = new BufferedOutputStream(audioOutputStream, BufferSize);
                                demuxer.setAudioOut(audioOut);
                            }
                        }
                    } else {
                        demuxer.printUsage();
                        return;
                    }
                    for (i = 0; i < demuxer.infiles.size(); i++) {
                        System.out.println(demuxer.infiles.get(i) + "\n");
                        demuxer.demuxFile((String) demuxer.infiles.get(i), demuxer, null);
                    }
                    demuxer.closeOutputStreams();
                    break;
                case 3: /* video and audio should be extracted */
                    if (demuxer.parseArg(args[0], 'i', demuxer.infiles)) {
                        if (demuxer.parseArg(args[1], 'v', outfilesVideo)) {
                            if (demuxer.parseArg(args[2], 'a', outfilesAudio)) {
                                videoFile = new File((String) outfilesVideo.get(0));
                                videoOutputStream = new FileOutputStream(videoFile);
                                videoOut = new BufferedOutputStream(videoOutputStream, BufferSize);
                                demuxer.setVideoOut(videoOut);
                                audioFile = new File((String) outfilesAudio.get(0));
                                audioOutputStream = new FileOutputStream(audioFile);
                                audioOut = new BufferedOutputStream(audioOutputStream, BufferSize);
                                demuxer.setAudioOut(audioOut);
                            } else {
                                demuxer.printUsage();
                                return;
                            }
                        }
                    } else {
                        demuxer.printUsage();
                        return;
                    }
                    for (i = 0; i < demuxer.infiles.size(); i++) {
                        System.out.println(demuxer.infiles.get(i) + "\n");
                        demuxer.demuxFile((String) demuxer.infiles.get(i), demuxer, null);
                        System.out.println("processed " + demuxer.packetCounter + " packets\n");
                        System.out.println("Video/Audio delay (ms):" + demuxer.videoAudioDelay());
                    }
                    break;
                default:
                    demuxer.printUsage();
            }
        } catch (java.io.IOException e) {
            System.out.println("Exception:" + e);
        }
    }
}