package tsStreamRipper;

import gui.MainWindow;

import java.io.File;
import java.io.IOException;
import java.util.Properties;
import java.util.logging.Logger;

import javazoom.jl.decoder.BitstreamException;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.OptionGroup;
import org.apache.commons.cli.Options;
import org.apache.commons.codec.DecoderException;

import tsStreamRipper.audio.ConverterOption;
import tsStreamRipper.audio.ConverterFactory.ConverterType;
import tsStreamRipper.dream.filestream.IFileStreamNameFormatter;
import tsStreamRipper.dream.filestream.InterpretTitelFormat;
import tsStreamRipper.dream.filestream.TitelInterpretFormat;
import tsStreamRipper.dream.webinterface.HttpParameter;
import tsStreamRipper.dream.webinterface.parser.EPGType;
import tsStreamRipper.dream.webstreaming.IExceptionCallback;
import tsStreamRipper.dream.webstreaming.StreamChannel;
import tsStreamRipper.util.logging.Config;

/**
 * CLI Interface zum Streamen eines Musik-Kanals
 */
public class StreamIt implements IExceptionCallback
{
    private static final Logger log = Logger.getLogger(StreamIt.class.getName());

    private static String host = null;
    private static int port = 8001;
    private static int restdauerOffset = 20;
    private static int webPort = 80;
    private static String channelId = null;
    private static String channel = null;
    private static String dest = null;
    private static String tmpDir = System.getProperty("java.io.tmpdir");
    private static String user = "";
    private static String pass = "";
    private static boolean verbose = false;
    private static ConverterOption convertOpt = null;
    private static EPGType epgType = EPGType.SKY_RADIO;

    private static IFileStreamNameFormatter nameFormat = null;
    private static boolean overwriteExisting = false;

    private static Properties channelIdList = new Properties();
    public static final String version = "tsStreamRipper Version 1.1.8";

    public static void main(String[] args) throws Exception
    {
        if (args.length == 0)
        {
            log.info("Starte tsStreamRipper GUI ...");
            MainWindow.showMainWindow();
            return;
        }

        // CLI Parameter auslesen ...
        Options opt = createCliOptions();

        try
        {
            CommandLineParser parser = new GnuParser();
            // parse the command line arguments
            CommandLine line = parser.parse(opt, args);

            if (line.hasOption("help"))
            {
                printHelp(opt);
                return;
            }

            if (line.hasOption("host"))
            {
                host = line.getOptionValue("host");
            }
            if (line.hasOption("streamport"))
            {
                port = Integer.parseInt(line.getOptionValue("streamport"));
            }
            if (line.hasOption("remainTimeOffset"))
            {
                restdauerOffset = Integer.parseInt(line.getOptionValue("remainTimeOffset"));
            }
            if (line.hasOption("webport"))
            {
                webPort = Integer.parseInt(line.getOptionValue("webport"));
            }
            if (line.hasOption("channelId"))
            {
                channelId = line.getOptionValue("channelId");
            }
            if (line.hasOption("channel"))
            {
                channel = line.getOptionValue("channel");
                channelIdList.load(StreamIt.class.getResourceAsStream("/channel.properties"));
                channelId = channelIdList.getProperty(channel);
                if (channelId == null)
                {
                    log.severe(String.format("Channel '%s' ist unbekannt.", channel));
                }
            }
            if (line.hasOption("channelKd"))
            {
                channel = line.getOptionValue("channel");
                channelIdList.load(StreamIt.class.getResourceAsStream("/channel_kd.properties"));
                channelId = channelIdList.getProperty(channel);
                if (channelId == null)
                {
                    log.severe(String.format("Channel '%s' ist unbekannt.", channel));
                }
            }
            if (line.hasOption("musicDir"))
            {
                dest = line.getOptionValue("musicDir");
            }
            if (line.hasOption("user"))
            {
                user = line.getOptionValue("user");
            }
            if (line.hasOption("pass"))
            {
                pass = line.getOptionValue("pass");
            }
            if (line.hasOption("tmpDir"))
            {
                tmpDir = line.getOptionValue("tmpDir");
            }
            if (line.hasOption("verbose"))
            {
                verbose = true;
            }
            if (line.hasOption("convert"))
            {
                boolean del = true;
                if (line.hasOption("keepOrig"))
                {
                    del = false;
                }
                try
                {
                    convertOpt = new ConverterOption(ConverterType.AAC, new File(line.getOptionValue("convert")), del);
                }
                catch (Exception e)
                {
                    log.warning(e.getMessage());
                    convertOpt = null;
                }
            }
            if (line.hasOption("convert_mp3"))
            {
                boolean del = true;
                if (line.hasOption("keepOrig"))
                {
                    del = false;
                }
                try
                {
                    convertOpt = new ConverterOption(ConverterType.MP3, new File(line.getOptionValue("convert_mp3")), del);
                }
                catch (Exception e)
                {
                    log.warning(e.getMessage());
                    convertOpt = null;
                }
            }
            if (line.hasOption("overwrite"))
            {
                overwriteExisting = true;
            }

            if (host == null)
            {
                log.severe("-host nicht gesetzt!");
                printHelp(opt);
                System.exit(1);
            }
            if (channelId == null)
            {
                log.severe("-channel, channelKd oder -channelId ist nicht gesetzt");
                printHelp(opt);
                System.exit(1);
            }

            // Filename Format
            if (line.hasOption("switchFilenameFormat"))
            {
                nameFormat = new InterpretTitelFormat();
            }
            else
            {
                // Default Title Format
                nameFormat = new TitelInterpretFormat();
            }

            // EPGType
            if (line.hasOption("epgType"))
            {
                String epgTypeStr = line.getOptionValue("epgType");
                if ("sky".equalsIgnoreCase(epgTypeStr))
                {
                    epgType = EPGType.SKY_RADIO;
                }
                else if ("swiss".equalsIgnoreCase(epgTypeStr))
                {
                    epgType = EPGType.SWISS_SATELLITE_RADIO;
                }
                else if ("kd".equalsIgnoreCase(epgTypeStr))
                {
                    epgType = EPGType.KD_23_5_GRAD;
                }
                else
                {
                    log.severe("Ungültiger epgType gesetzt : '" + epgTypeStr + "'");
                    printHelp(opt);
                    System.exit(1);
                }
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
            printHelp(opt);
        }

        // Logging konfigurieren
        Config.configureRuntimeLogging(verbose, false);

        log.info("Starte tsStreamRipper ... ");
        log.info(version);

        if (dest == null)
        {
            log.severe("'-musicDir' Verzeichnis wurde nicht angegeben oder existiert nicht. Programm endet jetzt");
            return;
        }

        log.info(String.format("Verbose : %s", Boolean.toString(verbose)));
        StreamIt si = new StreamIt();
        si.stream();
    }

    /**
     * Usage Informationen ausgeben
     * 
     * @param options
     */
    private static void printHelp(Options options)
    {
        System.out.println(version);
        HelpFormatter formatter = new HelpFormatter();
        formatter.setOptionComparator(new CliOptionComparator());
        formatter.printHelp("java -jar tsStreamRipper.jar -host <host> -musicDir <dir> [-channel <chan> | -channelKd <chan> | -channelId <id>] [options]", options);
    }

    private void stream() throws IOException, BitstreamException, DecoderException
    {
        File tempDir = null;
        File destDir = null;

        if (tmpDir != null)
            tempDir = new File(tmpDir);
        if (dest != null)
            destDir = new File(dest);

        if (tempDir != null)
        {
            if (!tempDir.exists())
            {
                log.severe(String.format("tmpDir '%s' existiert nicht, oder keine Berechtigung zum Zugriff.", tempDir.getAbsoluteFile()));
                System.exit(1);
            }
        }

        if (destDir != null)
        {
            if (!destDir.exists())
            {
                log.severe(String.format("'musicDir' Parameter fehlerhaft. Zielverzeichnis '%s' existiert nicht, oder keine Berechtigung zum Zugriff.", destDir.getAbsoluteFile()));
                System.exit(1);
            }
        }
        HttpParameter params = new HttpParameter(host, webPort, port, user, pass);
        
        Configuration conf;
        try
        {
            conf = new Configuration(params, channelId, tempDir, destDir, convertOpt, nameFormat, overwriteExisting, restdauerOffset, epgType);
            StreamChannel chan = new StreamChannel(this, conf);
            chan.start();
        }
        catch (Exception e)
        {
            e.printStackTrace();
            System.exit(1);
        }
    }

    @SuppressWarnings("static-access")
    private static Options createCliOptions()
    {
        Options options = new Options();

        // HTTP Optionen
        Option host = OptionBuilder.withArgName("host/ip").hasArg().withDescription("Dreambox Hostname oder IP Adresse").create("host");
        Option port = OptionBuilder.withArgName("portnumber").hasOptionalArg().withDescription("Dreambox Streaming-Portnummer. Default ist 8001.").create("streamport");
        Option webport_opt = OptionBuilder.withArgName("portnumber").hasOptionalArg().withDescription("Dreambox Webinterface-Portnummer. Default ist 80.").create("webport");
        Option user = OptionBuilder.withArgName("username").hasOptionalArg().withDescription("Falls HTTP Authentifizierung aktiv ist : Username").create("user");
        Option pass = OptionBuilder.withArgName("passwort").hasOptionalArg().withDescription("Falls HTTP Authentifizierung aktiv ist : Passwort").create("pass");

        // Streaming Channel Optionen
        Option channel = OptionBuilder
                .withArgName("name")
                .hasArg()
                .withDescription(
                        "SAT Musikkanal der Aufgezeichnet werden soll. Derzeit sind sind folgende SKY Kanaele vorkonfiguriert und werden als <name> Parameter akzeptiert:\r\n- 60er_70er\r\n- 80er_90er\r\n- country\r\n- deutsche_charts\r\n- lovesongs\r\n- rnb_hiphop\r\n- rock_hymnen")
                .create("channel");

        Option optChannelKd = OptionBuilder.withArgName("name").hasArg().withDescription("KD Musikkanal der Aufgezeichnet werden soll. Vorkonfiguriert wie SAT.").create(
                "channelKd");

        Option channelSref = OptionBuilder.withArgName("sRef").hasArg().withDescription(
                "Hier kann direkt die 'sRef' Referenz des Musikkanals angegeben werden. Die Parameter '-channelId' und '-channel' schliessen sich gegenseitig aus.").create(
                "channelId");

        // Dateiablage Optionen
        Option ziel = OptionBuilder.withArgName("dir").hasArg().withDescription("Verzeichnis in dem gerippte Musikstuecke abgelegt werden.").create("musicDir");
        Option tmpDir = OptionBuilder.withArgName("dir").hasOptionalArg().withDescription("Verzeichnis in dem temporaere Streams gespeichert werden.").create("tmpDir");

        // Verbose
        Option verbose = OptionBuilder.withDescription("Ausführliches Logging aktivieren").create("verbose");

        // Restdaueroffset
        Option remainOffset = OptionBuilder.withArgName("Sekunden").hasOptionalArg().withDescription(
                "Offset für gemeldete Restdauer des Liedende vom Web-Interface in Sekunden. Default ist +20 Sekunden.").create("remainTimeOffset");

        // EPG Auswahl, Default ist SKY Radio.
        Option epgTypeOpt = OptionBuilder.withArgName("sky|swiss|kd").hasOptionalArg().withDescription(
                "EPG Typ. Unterstützt derzeit : Sky Radio ('sky'), Swiss Satellite Radio ('swiss') und KD 23,5° ('kd'). Default ist 'sky'.").create("epgType");

        // Convert ?
        Option convert = OptionBuilder.withArgName("faacFile").hasArg().withDescription("Konvertierung MP2 Dateien nach m4a.").create("convert");
        Option convertmp3 = OptionBuilder.withArgName("lameFile").hasArg().withDescription("Konvertierung MP2 Dateien nach mp3.").create("convert_mp3");

        // Overwrite existing files ?
        Option overwrite = OptionBuilder.withDescription("Vorhandenen Titel überschreiben. Default : False").create("overwrite");

        Option help = OptionBuilder.withDescription("Dieser Hilfetext").create("help");

        Option delConvert = OptionBuilder.withDescription("Bei Konvertierung : Originale MP2-Datei behalten. Per Default werden diese gelöscht.").create("keepOrig");

        // FilenameOption
        Option filenameOption = OptionBuilder.withDescription("Musik-Dateiname im Format \"[Interpret] - [Titel]\" speichern.").create("switchFilenameFormat");

        options.addOption(host);
        options.addOption(port);
        options.addOption(webport_opt);
        options.addOption(ziel);
        options.addOption(tmpDir);
        options.addOption(user);
        options.addOption(pass);
        options.addOption(verbose);
        options.addOption(overwrite);
        options.addOption(delConvert);
        options.addOption(filenameOption);
        options.addOption(help);
        options.addOption(remainOffset);
        options.addOption(epgTypeOpt);

        // Convertierung: Entweder MP3 oder M4A
        OptionGroup convertGroup = new OptionGroup();
        convertGroup.addOption(convert);
        convertGroup.addOption(convertmp3);
        options.addOptionGroup(convertGroup);

        // Entweder eine sRef angeben, oder ein Alias, nicht beides
        OptionGroup chanGroup = new OptionGroup();
        chanGroup.addOption(channel);
        chanGroup.addOption(optChannelKd);
        chanGroup.addOption(channelSref);

        options.addOptionGroup(chanGroup);

        return options;
    }

    @Override
    public void exceptionOccured(Exception e)
    {
        log.severe("Das Streaming meldet einen Fehler : " + e.getMessage());
    }
}
