package tsStreamRipper.audio;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.List;
import java.util.logging.Logger;

/**
 * Basisklasse für die Umwandlung von MP2 zu einem Zielformat.
 */
public abstract class BaseConvert extends Thread implements IExternalEncoder
{
    private static final Logger log = Logger.getLogger(BaseConvert.class.getName());

    private boolean removeOrig = true;
    private File in = null;

    public BaseConvert(boolean removeOrig, File in)
    {
        this.removeOrig = removeOrig;
        this.in = in;
    }

    @Override
    public void startEncoding()
    {
        start();
    }

    protected abstract String getConverterName();

    protected abstract String getTargetFileFormat();

    @Override
    public void run()
    {
        preConvert();

        convert();

        postConvert();
    }

    /**
     * Konvertierungsprozess starten
     */
    private void convert()
    {
        StringBuilder sb = new StringBuilder();
        ProcessBuilder pb = new ProcessBuilder(getArgumentList());
        Process p = null;
        try
        {
            pb.redirectErrorStream(true);
            p = pb.start();

            BufferedReader isr = new BufferedReader(new InputStreamReader(p.getInputStream()));
            String line = null;
            while ((line = isr.readLine()) != null)
            {
                sb.append(line);
            }
            int exit = p.waitFor();

            if (exit != 0)
            {
                log.warning("Konvertierung ist möglicherweise fehlgeschlagen. Errorcode : " + exit);
                log.warning(String.format("Gesamte Programmausgabe : \r\n%s", sb.toString()));
            }
        }
        catch (IOException e)
        {
            log.throwing("BaseConvert", "run", e);
        }
        catch (InterruptedException e)
        {
            log.throwing("BaseConvert", "run", e);
        }
        finally
        {
            try
            {
                p.getInputStream().close();
            }
            catch (IOException e)
            {
            }
            try
            {
                p.getOutputStream().close();
            }
            catch (IOException e)
            {
            }
            try
            {
                p.getErrorStream().close();
            }
            catch (IOException e)
            {
            }
        }

        log.finer(String.format("Konvertierung zu %s beendet.", getTargetFileFormat()));
    }

    /**
     * Callback vor dem konvertieren einer Musikdatei
     */
    protected void preConvert()
    {

    }

    /**
     * Callback nach dem konvertieren einer Musikdatei
     */
    protected void postConvert()
    {
        if (removeOrig)
        {
            log.finer("Entferne Originaldatei " + in.getAbsolutePath());
            in.delete();
        }
        log.fine(String.format("Konvertierung nach %s beendet.", getTargetFileFormat()));
    }

    protected abstract List<String> getArgumentList();

    public boolean isRemoveOrig()
    {
        return removeOrig;
    }

    public File getInfile()
    {
        return in;
    }

}
