package tsStreamRipper.audio;

import java.io.File;
import java.io.IOException;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import javazoom.jl.converter.Converter;
import javazoom.jl.decoder.JavaLayerException;

/**
 * Diese Klasse konvertiert einee Mpeg2-Layer 1/2/3 Datei zu AAC. AAC Dateien lassen sich im iPad / iPod abspielen.
 */
public class ConvertMp2ToAAC extends BaseConvert
{
    private static final Logger log = Logger.getLogger(ConvertMp2ToAAC.class.getName());

    private File in = null;

    private File out = null;

    private File tmpWavFile = null;

    private File faac = null;

    private ID3 id3 = null;

    /**
     * 
     * @param in
     * @param tmpDir
     * @throws IOException
     */
    public ConvertMp2ToAAC(File in, File out, ID3 id3, File tmpDir, File faac, boolean removeOrig) throws IOException
    {
        super(removeOrig, in);
        this.in = in;
        this.out = out;
        this.tmpWavFile = File.createTempFile("tmpWav", null, tmpDir);
        this.faac = faac;
        this.id3 = id3;

        if (faac != null && !faac.exists())
        {
            throw new IOException("FAAC encoder path fehlerhaft : " + faac.getAbsolutePath());
        }
    }

    @Override
    protected void preConvert()
    {
        super.preConvert();
        Converter wavConv = new Converter();
        try
        {
            if (log.isLoggable(Level.FINE))
            {
                log.fine("Convertiere MP3 File " + in.getAbsolutePath());
                log.fine("  - WAV File " + tmpWavFile.getAbsolutePath());
                log.fine("  - MP4 File " + out.getAbsolutePath());
            }

            log.finer("Konvertiere zunächst zu einem WAV-File");
            wavConv.convert(in.getAbsolutePath(), tmpWavFile.getAbsolutePath());
        }
        catch (JavaLayerException e)
        {
            e.printStackTrace();
        }
    }

    @Override
    protected void postConvert()
    {
        if (log.isLoggable(Level.FINER))
        {
            log.finer("Lösche temporäre Wave Datei : " + tmpWavFile.getAbsolutePath());
        }
        tmpWavFile.delete();
        super.postConvert();
    }

    @Override
    protected List<String> getArgumentList()
    {
        LinkedList<String> args = new LinkedList<String>();

        args.add(faac.getAbsolutePath()); // Command
        args.add("-o");
        args.add(out.getAbsolutePath());
        args.add("--artist");
        args.add(id3.getInterpret());
        args.add("--title");
        args.add(id3.getTitel());
        args.add("--album");
        args.add(id3.getAlbum());
        args.add("--year");
        args.add(id3.getJahr());
        args.add(tmpWavFile.getAbsolutePath()); // InFile
        return args;
    }

    @Override
    protected String getConverterName()
    {
        return "FAAC";
    }

    @Override
    protected String getTargetFileFormat()
    {
        return "M4A";
    }

//    public static void main(String[] args) throws IOException
//    {
//
//        File in = new File("/temp/music/a.mp2");
//        File out = new File("/temp/music/a.m4a");
//
//        ID3 id3 = new ID3();
//        id3.setAlbum("AlbumTest");
//        id3.setInterpret("IntTest");
//        id3.setJahr("2010");
//        id3.setTitel("TestTitel");
//
//        ConvertMp2ToAAC c = new ConvertMp2ToAAC(in, out, id3, new File("/tmp"), new File("/usr/bin/faac"), false);
//        c.startEncoding();
//    }
}
