package tsStreamRipper.dream.filestream;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channel;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.WritableByteChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.jaudiotagger.audio.AudioFile;
import org.jaudiotagger.audio.AudioFileIO;
import org.jaudiotagger.audio.exceptions.CannotReadException;
import org.jaudiotagger.audio.exceptions.CannotWriteException;
import org.jaudiotagger.audio.exceptions.InvalidAudioFrameException;
import org.jaudiotagger.audio.exceptions.ReadOnlyFileException;
import org.jaudiotagger.tag.FieldKey;
import org.jaudiotagger.tag.Tag;
import org.jaudiotagger.tag.TagException;

import tsStreamRipper.audio.ID3;

/**
 * Hier wird die Musikdate verwaltet. Informationen über Titel, Interpret, Album und Jahr werden aus der EIT der Dreambox gelesen. Die ID3v2 Tag Aktualisierung erfolgt nachdem die
 * Datei in den Zielordner kopiert wurde.
 */
public class FileStream
{
    /**
     * Anzahl, wie oft eine Datei übersprungen wurde.
     */
    private static long skipCounter = 0;

    private ID3 id3 = new ID3();

    /**
     * Anzahl, wie oft eine doppelte Datei vorkommt
     */
    private static long skipDoubleCounter = 0;

    /**
     * Anzahl gerippter Dateien
     */
    private static long completeCounter = 0;

    private static final Logger log = Logger.getLogger(FileStream.class.getName());

    // Temporäre Datei 'stream...', dessen Streams
    // --------------------------------------------------------------------------------------------
    private FileOutputStream out = null;
    private FileChannel outChannel = null;
    private File outFile = null;
    // --------------------------------------------------------------------------------------------

    private File newFile = null;

    private boolean saveFile = true;

    private String skipReason = "";

    private File destinationDirectory = null;

    private boolean overwriteFiles = false;

    private IFileStreamNameFormatter fileNameStrategy = null;

    public FileStream()
    {
        this(null, null, new TitelInterpretFormat(), false);
    }

    public FileStream(File workDir, File destinationDirectory, IFileStreamNameFormatter filenameStrategy, boolean overwriteFiles)
    {
        this.destinationDirectory = destinationDirectory;
        this.fileNameStrategy = filenameStrategy;
        this.overwriteFiles = overwriteFiles;

        try
        {
            outFile = File.createTempFile("stream" + System.currentTimeMillis(), null, workDir);
            log.finer("create Temporary File : " + outFile.getAbsolutePath());
            out = new FileOutputStream(outFile);
            outChannel = out.getChannel();
        }
        catch (IOException e)
        {
            log.severe("Temporaere Datei kann nicht angelegt werden. Stoppe Aufnahme.");
        }
    }

    public OutputStream getOutputStream()
    {
        return out;
    }

    /**
     * Buffer darf nicht gefliped sein.
     * 
     * @param buffer
     */
    public void writeBuffer(ByteBuffer buffer)
    {
        buffer.flip();
        try
        {
            while (buffer.hasRemaining())
            {
                outChannel.write(buffer);
            }
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }
    }

    /**
     * File wird geschlossen. Temporäre Datei wird gelöscht.
     */
    public void closeFile()
    {
        try
        {
            out.close(); // Channel wird mit geschlossen
        }
        catch (IOException e)
        {
        }
        finally
        {
            try
            {
                String s = getNewFilename("mp3");

                // Prüfen, ob Datei bereits vorhanden ist
                checkFileExists();
                if (saveFile)
                {
                    newFile = new File(s);

                    log.info(String.format("Erstelle Musik-Datei : %s", s));

                    copyFile(outFile, newFile);

                    // IDv3 Tag dazuschreiben ...
                    if (log.isLoggable(Level.FINER))
                    {
                        log.finer("append ID3V2 Tag");
                    }

                    log.fine("Füge ID3 Tags zur Datei hinzu ...");

                    appendId3Tag(newFile);
                    completeCounter++;
                }
                else
                {
                    log.info(String.format("Ueberspringe Lied  : %s", s));
                    log.info(String.format("Grund : %s (%d / %d / %d)", skipReason, completeCounter, skipDoubleCounter, skipCounter));
                }
            }
            catch (Exception e)
            {
                log.severe("Fehler beim erstellen der Musikdatei aus dem Datenstrom.");
                log.throwing("FileStream", "closeFile", e);
            }
            finally
            {
                // Alte Datei löschen
                if (log.isLoggable(Level.FINER))
                {
                    log.finer("deleting old file : " + outFile.getAbsoluteFile());
                }
                outFile.delete();
            }
        }
    }

    /**
     * Überprüft ob eine MP2/MP3/M4A Variante des Lieds sich schon im Zielverzeichnis befindet.
     * 
     * @return
     */
    private boolean checkFileExists()
    {
        // mp2 Überprüfung
        String s = getNewFilename("mp2");
        File f = new File(s);

        if (f.exists())
        {
            if (saveFile && overwriteFiles)
            {
                log.finer("Lösche vorhandene Datei zwecks Überschreibung : " + s);
                f.delete();
                return false;
            }

            skipFileExists(String.format("Musikdatei '%s' existiert schon.", s));
            return true;
        }

        s = getNewFilename("mp3");
        f = new File(s);
        if (f.exists())
        {
            if (saveFile && overwriteFiles)
            {
                log.finer("Lösche vorhandene Datei zwecks Überschreibung : " + s);
                f.delete();
                return false;
            }

            skipFileExists(String.format("Musikdatei '%s' existiert schon.", s));
            return true;
        }

        s = getNewFilename("m4a");
        f = new File(s);
        if (f.exists())
        {
            if (saveFile && overwriteFiles)
            {
                log.finer("Lösche vorhandene Datei zwecks Überschreibung : " + s);
                f.delete();
                return false;
            }

            skipFileExists(String.format("Musikdatei '%s' existiert schon.", s));
            return true;
        }

        return false;
    }

    /**
     * Der neue Filename. Setzt sich zusammen aus "[Titel] - [Interpret].mp2". Enthält Titel oder Interpret ein Verzeichnistrennzeichen ist dieses zu escapen.
     * 
     * @return
     */
    private String getNewFilename(String extension)
    {
        String s = null;
        String dir = null;
        if (destinationDirectory != null)
        {
            dir = destinationDirectory.getAbsolutePath();
        }
        else
        {
            dir = outFile.getParent();
            if (dir == null)
            {
                dir = "/";
            }
        }
        // s = String.format("%s%sa.%s", dir, File.separator, extension);
        s = String.format("%s%s%s.%s", dir, File.separator, fileNameStrategy.getFilename(getStringFilename(getTitel()), getStringFilename(getInterpret()),
                getStringFilename(getAlbum()), getStringFilename(getJahr())), extension);
        return s;
    }

    private String getStringFilename(String source)
    {
        if (source == null)
        {
            return "NULL";
        }
        return source.replaceAll("[\\/:\"*?<>|]", "&").trim();
    }

    public String getInterpret()
    {
        return id3.getInterpret();
    }

    public void setInterpret(String interpret)
    {
        id3.setInterpret(interpret);
    }

    public String getTitel()
    {
        return id3.getTitel();
    }

    public void setTitel(String titel)
    {
        id3.setTitel(titel);
    }

    public String getAlbum()
    {
        return id3.getAlbum();
    }

    public void setAlbum(String album)
    {
        id3.setAlbum(album);
    }

    public String getJahr()
    {
        return id3.getJahr();
    }

    public void setJahr(String jahr)
    {
        id3.setJahr(jahr);
    }

    /**
     * Fügt ID3V2 Tag der Msuikdatei hinzu.
     * 
     * @param f
     *            Die Musikdatei
     */
    private void appendId3Tag(File f)
    {
        AudioFile audio;
        try
        {
            audio = AudioFileIO.read(f);
            Tag tag = audio.getTagOrCreateAndSetDefault();
            tag.setField(FieldKey.ARTIST, getInterpret());
            tag.setField(FieldKey.ALBUM, getAlbum());
            tag.setField(FieldKey.TITLE, getTitel());
            tag.setField(FieldKey.YEAR, getJahr());
            audio.commit();
        }
        catch (CannotReadException e)
        {
            e.printStackTrace();
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }
        catch (TagException e)
        {
            e.printStackTrace();
        }
        catch (ReadOnlyFileException e)
        {
            e.printStackTrace();
        }
        catch (InvalidAudioFrameException e)
        {
            e.printStackTrace();
        }
        catch (CannotWriteException e)
        {
            e.printStackTrace();
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }

    /**
     * Die Datei wird übersprungen, weil sie bereits existiert.
     */
    private void skipFileExists(String skipReason)
    {
        skipDoubleCounter++;
        this.saveFile = false;
        this.skipReason = skipReason;
    }

    /**
     * Die Datei wird übersprungen, weil sie nicht korrekt aufgenommen werden konnte.
     */
    public void skipFileError(String skipReason)
    {
        skipCounter++;
        this.saveFile = false;
        this.skipReason = skipReason;
    }

    /**
     * Wird die Datei gespeichert?
     * 
     * @return
     */
    public boolean isSaveFile()
    {
        return saveFile;
    }

    @Override
    public String toString()
    {
        String.format("Titel: '%s'  Interpret : '%s'  Album : '%s'", getTitel(), getInterpret(), getAlbum());
        return super.toString();
    }

    /**
     * Das moven von Dateien im Unix-Filesystem über Devicegrenzen hinweg ist teilweise buggy. Deswegen kopieren wir die Datei
     * 
     * @param fis
     * @param os
     * @throws FileNotFoundException
     */
    public void copyFile(File source, File destination)
    {
        FileChannel fileChannel = null;
        WritableByteChannel wc = null;
        try
        {
            FileInputStream fis = new FileInputStream(source);
            FileOutputStream fos = new FileOutputStream(destination);
            fileChannel = fis.getChannel();

            wc = Channels.newChannel(fos);
            fileChannel.transferTo(0, fileChannel.size(), wc);
        }
        catch (Exception e)
        {
            log.severe("Fehler beim kopieren der Musikdatei in das Zielverzeichnis. Dateirechte prüfen.");
            log.throwing(getClass().getName(), "copyFile", e);
        }
        finally
        {
            closeChannel(fileChannel);
            closeChannel(wc);
        }
    }

    private void closeChannel(Channel c)
    {
        try
        {
            c.close();
        }
        catch (IOException e)
        {
            log.throwing("FileStream", "closeChannel", e);
        }
    }

    public boolean equals(Object obj)
    {
        return id3.equals(obj);
    }

    public int hashCode()
    {
        return id3.hashCode();
    }

    public ID3 getId3()
    {
        return id3;
    }

    public File getNewFile()
    {
        return newFile;
    }
}
