package tsStreamRipper.dream.webinterface;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.UnknownHostException;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.apache.http.HttpEntity;
import org.xml.sax.helpers.DefaultHandler;

import tsStreamRipper.dream.webstreaming.IExceptionCallback;
import tsStreamRipper.dream.webstreaming.StreamBase;

/**
 * Abstrakte Basisklasse für Dreambox Web-Befehle.
 */
public abstract class WebCommand extends StreamBase
{
    private static final Logger log = Logger.getLogger(WebCommand.class.getName());

    private String url = null;
    
    private boolean commandCompleted = false;

    public WebCommand(HttpParameter httpParameter, IExceptionCallback exceptionCallback)
    {
        super(httpParameter, exceptionCallback);
    }

    
    public boolean isCommandCompleted()
    {
        return commandCompleted;
    }


    @Override
    protected String getUrl()
    {
        return url;
    }

    @Override
    public void run()
    {
        this.url = String.format("http://%s:%d/web/%s", getHttpParameter().getHost(), getHttpParameter().getWebPort() , getCommandName());
                
        
        if (log.isLoggable(Level.FINE))
            log.fine("WebInterface Zugriff : " + this.url);

        try
        {
            HttpEntity entity = getEntity();
            SAXParserFactory factory = SAXParserFactory.newInstance();
            SAXParser saxParser = factory.newSAXParser();
            if (doDebugContent())
            {
                String s = debugContent(entity.getContent());
                saxParser.parse(s, getDocumentParser());
            }
            else
            {
                saxParser.parse(entity.getContent(), getDocumentParser());
            }
            if (log.isLoggable(Level.INFO))
                log.finer("WEB-COMMAND : " + getCommandName() + "finished.");
            
            commandCompleted = true;

        }
        catch (UnknownHostException e)
        {
            log.severe(String.format("Dreambox '%s:%d' ist nicht erreichbar.", getHttpParameter().getHost(),
                    getHttpParameter().getWebPort()));
        }
        catch (CommandException e)
        {
            log.severe(e.getMessage());
            throw e;
        }
        catch (Exception e)
        {
            commandCompleted = false;
            e.printStackTrace();
        }
        finally
        {
            // Benachrichtigung, das wir fertig sind ...
            synchronized (this)
            {
                if (sleepTimeBeforeNotify() > 0)
                {
                    try
                    {
                        sleep(sleepTimeBeforeNotify());
                    }
                    catch (InterruptedException e)
                    {
                        // Hier sollten wir nicht hinkommen ...
                        e.printStackTrace();
                    }
                }
                notifyAll();
            }
        }
    }

    private String debugContent(InputStream is)
    {
        InputStreamReader isr = new InputStreamReader(is);
        BufferedReader br = new BufferedReader(isr);

        String s = null;
        StringBuilder sb = new StringBuilder();
        do
        {
            try
            {
                s = br.readLine();
            }
            catch (IOException e)
            {
                e.printStackTrace();
            }
            sb.append(s);
        }
        while (s != null);
        return sb.toString();
    }

    /**
     * Anzahl der Sekunden, die gewartet werden soll, bis der Aufrufer über Notify über die Aktion benachrichtigt wird
     * 
     * @return Millisekunden
     */
    protected long sleepTimeBeforeNotify()
    {
        return 0;
    }

    /**
     * WebService Command
     * 
     * @return
     */
    abstract protected String getCommandName();

    abstract protected DefaultHandler getDocumentParser();

    protected boolean doDebugContent()
    {
        return false;
    }
}
