package tsStreamRipper.dream.webstreaming;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.UnknownHostException;
import java.nio.channels.FileChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.ConnectionClosedException;
import org.apache.http.HttpEntity;
import org.apache.http.client.ClientProtocolException;

import tsStreamRipper.Configuration;
import tsStreamRipper.audio.ConverterFactory;
import tsStreamRipper.audio.ConverterFactory.ConverterType;
import tsStreamRipper.audio.IExternalEncoder;
import tsStreamRipper.dream.filestream.FileStream;
import tsStreamRipper.dream.webinterface.CheckStandbyCommand;
import tsStreamRipper.dream.webinterface.HttpParameter;
import tsStreamRipper.dream.webinterface.StreamInfo;
import tsStreamRipper.dream.webinterface.WebCommand;
import tsStreamRipper.dream.webinterface.ZapCommand;
import tsStreamRipper.mp3.SilenceRecognitionStream;
import tsStreamRipper.mpeg.MPEG2TransportStreamDemultiplexer;
import tsStreamRipper.mpeg.MPEG2TransportStreamReader;
import tsStreamRipper.util.ByteBufferOutputStream;

/**
 * Diese Klasse überwacht einen Transportstrom von der Dreambox und erstellt die Musikstücke.
 * 
 * Der Transportstrom, den die Dreambox ausgeliefert hat keinen Elementary Stream mit Event Information Table Einträgen,
 * deshalb werden diese über das DM Webinterface geholt.
 */
public class StreamChannel extends StreamBase implements IDurationCallback, IExceptionCallback
{
    private static final Logger log = Logger.getLogger(StreamChannel.class.getName());

    private boolean audioStreamExists = false;

    private Configuration config = null;

    /**
     * URL des Streams
     */
    private String url = null;

    private FileStream fs = null;

    private long startSilenceDetectionTime = Long.MAX_VALUE;

    private long endSilenceDetectionTime = Long.MAX_VALUE;

    private boolean debugBuffer = false;
    
    
    // 1 MB Default-Buffer ist ausreichend für 10 Sekunden
    private ByteBufferOutputStream bos = new ByteBufferOutputStream();

    /**
     * Stillererkennung nciht geklappt oder erstes Musikstück
     */
    private boolean firstSample = true;

    /**
     * Anzahl der MS die vor und nach Ende des Musikstücks gemäß EPG Stille erkannt wird.
     */
    private long silenceRecognitionOffset = 10000;

    @Override
    public void run()
    {
        setName("StreamChannel : " + config.getChannel());
        boolean startNewStream = true;

        boolean aquireStreamInfoStarted = false;

        long startAquireStreamInfoOffset = 10000; // 10 Sekunden nach Anfang der Aufnahme EIT auslesen

        try
        {
            zapToChannel();
            
            HttpEntity entity = getEntity();
            if (entity != null)
            {

                MPEG2TransportStreamDemultiplexer decoder = new MPEG2TransportStreamDemultiplexer(
                        new BufferedOutputStream(bos));
                // Dekodierte MP3 Daten im Puffer halten zwecks Stilleerkennung.
                MPEG2TransportStreamReader reader = new MPEG2TransportStreamReader(entity.getContent(), decoder);

                long startGetInfo = 0;

                // Schleife : Transport-Stream auslesen ...
                do
                {
                    if (audioStreamExists)
                    {
                        if (startNewStream)
                        {
                            if (fs != null)
                            {
                                fs.closeFile();
                                if (config.getConvOpt() != null && fs.isSaveFile())
                                {
                                    convertMp2();
                                }
                            }

                            fs = new FileStream(config.getTmpDir(), config.getDestination(), config.getNameFormatter(), config.isOverwriteFiles());
                            if (firstSample)
                            {
                                firstSample = false;
                                fs
                                        .skipFileError("Fehlerhafte Stilleerkennung oder erstes Musikstück nach Programmstart.");
                            }
                            startNewStream = false;
                            aquireStreamInfoStarted = false;
                            startSilenceDetectionTime = Long.MAX_VALUE;
                            endSilenceDetectionTime = Long.MAX_VALUE;
                            startGetInfo = System.currentTimeMillis() + startAquireStreamInfoOffset;
                        }

                        //
                        // Laufzeit/Interpret des aktuellen Lieds von der Dreambox holen. EIT Info's sind im TS-Stream
                        // nicht vorhanden !
                        //
                        if (!aquireStreamInfoStarted && (System.currentTimeMillis() > startGetInfo))
                        {
                            log.finer("Getting Stream information : " + config.getEpgType());
                            aquireStreamInfoStarted = true;

                            // Derzeit wird nur das EPG von SKY Radio unterstützt
                            HttpParameter epgParam = getHttpParameter();
                            StreamInfo si = new StreamInfo(epgParam, this, this, config.getEpgType().getEpgParser(fs));
                            si.start();
                        }

                        if (System.currentTimeMillis() < startSilenceDetectionTime)
                        {
                            // Noch keine Ruheerkennung ... Puffer wegschreiben
                            fs.writeBuffer(bos.getBuffer());
                            // Puffer löschen
                            bos.clear();
                        }

                        if (System.currentTimeMillis() > endSilenceDetectionTime)
                        {
                            SilenceRecognitionStream srs = new SilenceRecognitionStream(bos.getBuffer());

                            // Ungethreaded starten ... wenn's klappt ist besser, sonst müssen wir syncen ...
                            srs.run();
                            if (srs.hasSilence())
                            {
                                fs.writeBuffer(srs.getLastSongEnd());
                                // Neuen Anfang erstellen
                                bos.getBuffer().compact();
                                if (log.isLoggable(Level.FINER))
                                    log.finer(String.format("current position in byteBuffer : %d ", bos.getBuffer()
                                            .position()));
                            }
                            else
                            {
                                if (debugBuffer)
                                {
                                    bos.getBuffer().position(0);
                                    FileChannel fileChannel = new FileOutputStream(File.createTempFile("tmpBuf"+System.currentTimeMillis(),null)).getChannel();
                                    fileChannel.write(bos.getBuffer());
                                    fileChannel.close();
                                }
                                // Ruhe wurde nicht erkannt. Puffer ist somit uninteressant
                                bos.getBuffer().clear();
                                log.finer("Ruhe in Musikstück nicht erkannt.");

                                // Wenn Ruhe nicht erkannt wurde, dann ist auch das darauf folgende Lied kaputt.
                                firstSample = true;
                                fs
                                        .skipFileError("Stille zwischen dem aufgenommenen und dem darauf folgendem Musikstueck wurde nicht erkannt.");
                            }
                            startNewStream = true;
                        }
                    }
                    else
                    {
                        if (decoder.getAudioElementaryStream() != null)
                        {
                            audioStreamExists = true;
                            log.finer("AudioChannel innerhalb Transportstrom vorhanden ...");
                        }
                    }
                }
                // loop: Falls Stille erkannt wurde, nicht das nächste Packet dekodieren.
                while (reader.decodeNextFrame() && !isThreadStopping());
                log.fine("Streaming bricht ab ...");
            }
            else
            {
                log.warning("Verbindungsaufbau nicht möglich ...");
                throw new ConnectionClosedException("Verbindungsaufbau nicht möglich.");
            }
        }
        catch (UnknownHostException e)
        {
            log.severe(String.format("Der Host '%s:%d' bietet keinen Musik-Stream an.", getHttpParameter().getHost(),
                    getHttpParameter().getPort()));
        }
        catch (ConnectionClosedException e)
        {
            System.out.println("Verbindung zum Server verloren.");
        }
        catch (ClientProtocolException e)
        {
            e.printStackTrace();
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
        finally
        {
            log.info(String.format("Streaming der ChannelID %s endet jetzt.", config.getChannel()));
        }
    }

    private void convertMp2() throws IOException
    {
        IExternalEncoder encoder = ConverterFactory.getEncoder(config.getConvOpt(),fs.getNewFile(), fs.getId3(), config.getTmpDir());
        encoder.startEncoding();
    }

    
    private void waitForCompletion(WebCommand wc, String errorText)
    {
        try
        {
            synchronized (wc)
            {
                wc.wait();
                if (!wc.isCommandCompleted())
                {
                    throw new RuntimeException(errorText);
                }
            }
        }
        catch (InterruptedException e1)
        {
            interrupt();
            e1.printStackTrace();
        }
    }
    
    /**
     * Auf den mitzuschneidenen Kanal wechseln wegen EPG Info ...
     */
    private void zapToChannel()
    {
        // Wenn die Box im Standy ist, dann kann nicht per Webinterface der Kanal gewechselt werden.
        CheckStandbyCommand sb = new CheckStandbyCommand(getHttpParameter(), this);
        log.finer("checke PowerState der Dreambox ...");
        
        sb.start();
        waitForCompletion(sb, "Checken des Power-States klappt nicht ...");

        // Falls Box im Standby ist, dann Online gehen ...
        if (sb.getInStandby())
        {
            log.info("Dreambox ist im Standby. Aufwachen ...");
            sb = new CheckStandbyCommand(getHttpParameter(), this, true);
            sb.start();
            waitForCompletion(sb, "Aufwachen der Dreambox klappt nicht ...");
        }
        
        // Auf Kanal umschalten ...
        ZapCommand zap = new ZapCommand(getHttpParameter(), this, config.getChannel());
        zap.start();
        waitForCompletion(zap, "Umschalten auf den aufzunehmenden Kanal klappt nicht ...");
    }

    public StreamChannel(IExceptionCallback exceptionCallback, Configuration config) throws FileNotFoundException
    {
        super(config.getHttpParameter(), exceptionCallback);
        this.config = config;
        this.url = String.format("http://%s:%d/%s", config.getHttpParameter().getHost(), config.getHttpParameter().getPort(), config.getChannel());
        printInfo();
    }

    private void printInfo()
    {
        log.info(String.format("Stream wird geholt von : http://%s:%d", config.getHttpParameter().getHost(), config.getHttpParameter().getPort()));
        log.info(String.format("ChannelId des Streams : %s", config.getChannel()));
        log.info(String.format("EPG Type des Streams : %s", config.getEpgType()));
        log.info(String.format("Vollstaendig gestreamte MP2-Dateien werden in das Verzeichnis '%s' kopiert.", config.getDestination().getAbsolutePath()));
        if (config.getConvOpt() != null)
        {
            if (config.getConvOpt().getType().equals(ConverterType.AAC))
            {
                log.info("Konvertierung zu m4a : Ja.");
                log.info(String.format("Pfad auf faac-encoder : '%s'", config.getConvOpt().getConverterPath().getAbsolutePath()));
                log.info(String.format("Loeschen orginaler MP2-Dateien : %s", Boolean.valueOf(config.getConvOpt().isRemoveOriginalFile())));
            }
            else if (config.getConvOpt().getType().equals(ConverterType.MP3))
            {
                log.info("Konvertierung zu MP3-Datei : Ja.");
                log.info(String.format("Pfad auf lame-encoder : '%s'", config.getConvOpt().getConverterPath().getAbsolutePath()));
                log.info(String.format("Loeschen orginaler MP2-Dateien : %s", Boolean.valueOf(config.getConvOpt().isRemoveOriginalFile())));
            }
            else
            {
                log.severe("Konvertierungstyp unbekannt : " + config.getConvOpt().getType().toString());
            }
        }
        else
        {
            log.info("Konvertierung zu m4a oder mp3 : Nein");
        }
        log.info(String.format("Restdauerkorrektur für Musikende in Sekunden : %s", config.getRestdauerKorrektur()));
        log.info("Überschreibe existierende Dateien : " + Boolean.toString(config.isOverwriteFiles()));
        log.info(String.format("Temporaere Dateien werden im Verzeichnis '%s' abgelegt", config.getTmpDir().getAbsolutePath()));
        log.info(config.getNameFormatter().info());
        log.info("Falls eine Datei nicht erstellt wird, werden Statusinformation ueber die Anzahl der (gerippten/doppelten/beschaedigten) Dateien"
                + " seit Programmstart angegeben.");
    }

    @Override
    protected String getUrl()
    {
        return url;
    }

    /**
     * Die Stilleerkennung wird erst <code>silenceRecognitionStartOffset</code> (default 6) Sekunden vor dem Ende laut
     * EPG aktiviert und bis <code>silenceRecognitionEndOffset</code> (defualt 6) Sekunden nach Ende als gültig
     * akzeptiert. Wird innerhalb dieses Fensters keine Stille erkannt, wird das Musikstück als fehlerhaft markiert.
     */
    @Override
    public void setDuration(int sec)
    {
        // Laufzeit der aktuellen Lieds. Stilleerkennung ca. 10 Sekunden vorher starten
        if (log.isLoggable(Level.FINER))
            log.finer("Restlauzeit des aktuellen Musikstücks in Sekunden : " + sec);

        // Früher musste (sec - 2) das Liedende vorgezogen werden ... Das führt nun zu Fehlern !
        // Aktuell sendet Sky +20 Sekunden ?! 
        // Offset ist nun konfigurierbar.
        long endeNachEpg = System.currentTimeMillis() + (sec + config.getRestdauerKorrektur()) * 1000;

        startSilenceDetectionTime = endeNachEpg - silenceRecognitionOffset;
        endSilenceDetectionTime = endeNachEpg + silenceRecognitionOffset;
    }

    @Override
    public void exceptionOccured(Exception e)
    {
        log.severe("Verbindung zur Dreambox vermutlich gestört.");
        stopStreamBaseThread();
    }
}
