package tsStreamRipper.mp3;

import java.nio.ByteBuffer;
import java.util.logging.Level;
import java.util.logging.Logger;

import javazoom.jl.decoder.Bitstream;
import javazoom.jl.decoder.Decoder;
import javazoom.jl.decoder.Header;
import tsStreamRipper.util.ByteBufferInputStream;

/**
 * Dieser Thread überwacht den MP2/3 Output Stream und meldet eine Stille, falls erkannt und terminiert anschlißend.
 * 
 * Update : Wir versuchen es zunächst ohne Thread. Sollte die Stilleerkennung zu lange dauern, wir also den InputStream
 * nicht schnell genug auslesen können, müssen wir threaden.
 * 
 * Vorgehen:<br>
 * <br>
 * Diese Stilleerkennung versucht iterativ "Stille" zu erkennen. Dabei wird der Schwellwert, ab wann "Stille" erkannt
 * wird mit jedem Durchlauf um 10 Amplitudenpunkte erhöht. Die Anzahl der Samples wird alle 2 Durchläufe um 1 reduziert.
 * 
 * Wird Stille erkannt, enthält <code>lastSongEnd</code> einen ByteBuffer der zum auslesen noch gefliped werden muss.<br>
 * <br>
 * Der aktuzelle Puffer wird an der "Stilleposition" gelasse und kann durch den Aufruf von compact() als Puffer für das
 * neue Lied wiederverwendet werden.
 */
public class SilenceRecognitionStream extends Thread
{
    private static final Logger log = Logger.getLogger(SilenceRecognitionStream.class.getName());

    private int schwelle = 10; // 60

    private int samples = 11; // 7

    private ByteBuffer byteBuffer = null;

    private ByteBuffer lastSongEnd = null;

    private boolean hasSilence = false;

    private int maxPass = 10;

    public SilenceRecognitionStream(ByteBuffer buffer)
    {
        this.byteBuffer = buffer;
        this.byteBuffer.flip();

        setName("SilenceRecognition Thread");
    }

    @Override
    public void run()
    {
        long startTime = System.currentTimeMillis();
        int pass = 0;

        if (log.isLoggable(Level.FINE))
            log.fine("SilenceBuffer limit  : " + byteBuffer.limit());

        while (!hasSilence && (pass < maxPass))
        {
            byteBuffer.position(0);

            Bitstream stream = new Bitstream(new ByteBufferInputStream(byteBuffer));
            Decoder decoder = new Decoder();
            SilenceRecognitionBuffer buffer = new SilenceRecognitionBuffer(schwelle, samples);
            decoder.setOutputBuffer(buffer);

            while (!hasSilence)
            {
                try
                {
                    Header header = stream.readFrame();

                    if (header == null)
                    {
                        // Stream Ende ...
                        log.finer("Buffer bis zum Ende gelesen.");
                        break;
                    }
                    buffer = (SilenceRecognitionBuffer) decoder.decodeFrame(header, stream);
                    stream.closeFrame();

                    if (buffer.isSilence())
                    {
                        log.finer("Silence detected");

                        if (log.isLoggable(Level.FINER))
                            log.finer(String.format("Silence-End Position : %d ", byteBuffer.position()));

                        // Puffer für das Liedende erstellen. Dieser darf nicht gefliped sein!
                        lastSongEnd = byteBuffer.duplicate();
                        lastSongEnd.position(byteBuffer.position()).limit(byteBuffer.position());

                        if (log.isLoggable(Level.FINER))
                            log.finer(String.format("Last Song cutPosition : %d ", lastSongEnd.limit()));
                        if (log.isLoggable(Level.FINE))
                            log.fine(String.format("Pass : %d , Schwelle : %d , Samples : %d, Position : %d ", pass,
                                    schwelle, samples, byteBuffer.position()));
                        hasSilence = true;
                    }
                }
                catch (Exception e)
                {
                    log
                            .severe("Stilleerkennung: Fehler beim dekodieren des Streams. Möglicherweise ist die SAT-Schüssel schlecht ausgerichtet oder es liegt Schnee drauf.");
                    break;
                }
            }

            schwelle += 10;

            // samples jeden 2. Durchgang reduzieren. Bei 8 Durchläufen wäre das 4 mal
            if (pass % 2 == 1)
                samples -= 1;
            samples = Math.max(samples, 6); // Unter 6 Samples wird's eng
            pass++;

            if (log.isLoggable(Level.FINE))
                log.fine(String.format(
                        "Nächster Stille-Test mit folgenden Parametern : Pass '%d', Schwelle '%d', Samples '%d'", pass,
                        schwelle, samples));
        }

        long endTime = System.currentTimeMillis();
        long duration = endTime - startTime;
        if (log.isLoggable(Level.FINE))
            log.fine(String.format("SilenceDetectionTime %d [ms]", duration));
        if (duration > 2000)
        {
            log.warning("Stilleerknnung läuft sehr lange : " + duration + " [ms].");
        }

    }

    /**
     * Buffer ist nicht geflipped.
     * 
     * @return
     */
    public ByteBuffer getLastSongEnd()
    {
        return lastSongEnd;
    }

    public boolean hasSilence()
    {
        return hasSilence;
    }
}
