package tsStreamRipper.mpeg;

import tsStreamRipper.util.UnsignedByteArray;

/**
 * This Class decodes a TransportPacket Header.
 */
public class MPEG2TransportPacketHeader
{

    public enum TransportPacketType
    {
        PACKETIZED_ELEMENTRY_STREAM,
        ADAPTION_FIELD,
        ADAPTION_FIELD_AND_PACKETIZED_ELEMENTRY_STREAM,
        SECTION,
        UNKNOWN
    };

    private UnsignedByteArray ba = null;

    public MPEG2TransportPacketHeader()
    {
    }

    /**
     * 
     * @param header
     */
    public MPEG2TransportPacketHeader(UnsignedByteArray ba)
    {
        this.ba = ba;
    }

    private boolean isBitSet(int bitmap, int bitnummer)
    {
        int base = 1;
        for (int i = 0; i < bitnummer; i++)
        {
            base = base << 1;
        }
        return (bitmap & base) != 0;
    }

    public String info()
    {
        StringBuilder sb = new StringBuilder();
        sb.append("SYNC      : ").append(getSyncByteExists()).append("\r\n");
        sb.append("TRANERR   : ").append(getTransportError()).append("\r\n");
        sb.append("ADAPFIELD : ").append(getAdaptionFieldExists()).append("\r\n");
        sb.append("PAYSTART  : ").append(getPayloadStart()).append("\r\n");
        sb.append("PAYEXISTS : ").append(getPayloadExists()).append("\r\n");
        sb.append("SCRAM      : ").append(getScrambeled());
        return sb.toString();
    }

    public boolean getSyncByteExists()
    {
        return ba.valueAt(0) == 0x47;
    }

    public boolean getTransportError()
    {
        return isBitSet(ba.valueAt(1), 7);
    }

    public boolean getAdaptionFieldExists()
    {
        return isBitSet(ba.valueAt(3), 5);
    }

    public boolean getPayloadExists()
    {
        return isBitSet(ba.valueAt(3), 4);
    }

    public boolean getPayloadStart()
    {
        return isBitSet(ba.valueAt(1), 6);
    }

    public boolean getScrambeled()
    {
        return (!isBitSet(ba.valueAt(3), 6)) && (!isBitSet(ba.valueAt(3), 7));
    }

    public int getPID()
    {
        int l = ba.valueAt(1) & 31 << 8;
        l += ba.valueAt(2);
        return l;
    }

    public TransportPacketType getTransportPacketType()
    {

        if (getAdaptionFieldExists() && getPayloadExists())
        {
            // Code cannot handle that yet !
            return TransportPacketType.ADAPTION_FIELD_AND_PACKETIZED_ELEMENTRY_STREAM;
        }
        if (getAdaptionFieldExists())
        {
            return TransportPacketType.ADAPTION_FIELD;
        }
        else if (getPayloadExists())
        {
            return TransportPacketType.PACKETIZED_ELEMENTRY_STREAM;
        }
        return TransportPacketType.UNKNOWN;
    }

    @Override
    public String toString()
    {
        return info();
    }
}
