package tsStreamRipper.mpeg;

import java.io.IOException;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import tsStreamRipper.mpeg.elementarystream.ElementaryStream;
import tsStreamRipper.mpeg.elementarystream.ElementaryStream.StreamType;
import tsStreamRipper.mpeg.elementarystream.IgnoreElementaryStream;
import tsStreamRipper.mpeg.elementarystream.OutputElementaryStream;
import tsStreamRipper.mpeg.elementarystream.PatElementaryStream;
import tsStreamRipper.util.UnsignedByteArray;

/**
 * This class decodes Packets received from MPEG2TransportStreamReader. This class is not Thread safe! <br>
 * ISO/IEC 13818-1 : 2000 - Section 2.4.3.3<br>
 * <br>
 * PID:<br>
 * 0x0000 Program Association Table<br>
 * 0x0001 Conditional Access Table<br>
 * 0x0002 Transport Stream Description Table<br>
 * 0x0003-0x000F Reserved<br>
 * 0x00010 … 0x1FFE May be assigned as network_PID, Program_map_PID, elementary_PID, or for other purposes<br>
 * 0x1FFF Null packet NOTE – The transport packets with PID values
 * 
 * 
 */
public class MPEG2TransportStreamDemultiplexer implements ITransportPacketListener
{
    private static final Logger log = Logger.getLogger(MPEG2TransportStreamDemultiplexer.class.getName());

    private UnsignedByteArray tsFrame = new UnsignedByteArray();
    private MPEG2TransportPacketHeader mpegHeader = new MPEG2TransportPacketHeader(tsFrame);
    private Map<Long, ElementaryStream> streams = new HashMap<Long, ElementaryStream>();

    private OutputStream audioOut = null;

    public MPEG2TransportStreamDemultiplexer(OutputStream audio)
    {
        audioOut = audio;
    }

    @Override
    public void transportPackedReceived(byte[] tsPacket) throws IOException
    {
        tsFrame.setWrappedByteArray(tsPacket);
        int pid = mpegHeader.getPID();
        boolean payLoadStart = mpegHeader.getPayloadStart();
        if (pid == 0x1FFF)
        {
            return; // Ignore NULL Packets.
        }

        ElementaryStream esHandler = null;
        switch (mpegHeader.getTransportPacketType())
        {
            case PACKETIZED_ELEMENTRY_STREAM:
                esHandler = getHandler(pid, tsPacket);
                if (esHandler != null)
                {
                    esHandler.handlePacket(getPesData(tsPacket, 0), payLoadStart);
                }
                break;
            case ADAPTION_FIELD_AND_PACKETIZED_ELEMENTRY_STREAM:
                esHandler = getHandler(pid, tsPacket);
                if (esHandler != null)
                {
                    byte[] adaptionField = getAdaptionField();
                    esHandler.handleAdaptionField(adaptionField);
                    esHandler.handlePacket(getPesData(tsPacket, adaptionField.length), payLoadStart);
                }
                break;
            case SECTION:

                break;
            case ADAPTION_FIELD:
                break;
            default:
                break;
        }
    }

    private byte[] getAdaptionField()
    {
        int adaptionFieldSize = tsFrame.valueAt(4) + 1; // Das 5te Byte ist die AdationFieldSize. Das Size-Byte kopieren wir mit (deshalb +1)

        byte[] b = new byte[adaptionFieldSize];
        System.arraycopy(tsFrame.getWrappedData(), 4, b, 0, adaptionFieldSize);

        return b;
    }

    private ElementaryStream getHandler(long pid, byte[] tsPacket)
    {
        ElementaryStream handler = streams.get(pid);
        if (handler == null)
        {
            if (mpegHeader.getPayloadStart())
            {
                MPEG2PESHeader pesHeader = new MPEG2PESHeader(getPesDataHeader(tsPacket));

                if (pid == 0)
                {
                    handler = new PatElementaryStream(pid);
                }
                else if (pesHeader.isAudioPacket() && pesHeader.hasPTS())
                {
                    handler = new OutputElementaryStream(pid, StreamType.AUDIO, audioOut);

                }
                else
                {
                    handler = new IgnoreElementaryStream(pid);
                }
                streams.put(pid, handler);

                if (log.isLoggable(Level.FINE))
                {
                    if (pid == 0)
                    {
                        log.fine("PAT-Stream auf pid 0 vorhanden ...");
                    }
                    else
                    {
                        log.fine(String.format("Elementary Stream auf pid %s vom Type %s", pid, pesHeader.info()));
                    }
                }

                return handler;
            }
            else
            {
                if (log.isLoggable(Level.FINER))
                    log.finer("Ueberspringe Transport Packet bis payload beginnt auf PID : " + pid);
            }
            return null;
        }
        else
        {
            return handler;
        }
    }

    public OutputElementaryStream getAudioElementaryStream()
    {
        for (ElementaryStream es : streams.values())
        {
            if (es.getType().equals(StreamType.AUDIO))
            {
                return (OutputElementaryStream) es;
            }
        }
        return null;
    }

    /**
     * Returns PES payload
     * 
     * @param tsFrame
     * @return
     */
    private byte[] getPesData(byte[] tsFrame, int adaptionFieldSize)
    {
        int pesSegmentSize = MPEG2TransportStreamReader.TRANSPORT_PACKET_SIZE - MPEG2TransportStreamReader.TRANSPORT_PACKET_HEADER_SIZE - adaptionFieldSize; // 188 - 4 header bytes

        byte[] b = new byte[pesSegmentSize];
        System.arraycopy(tsFrame, MPEG2TransportStreamReader.TRANSPORT_PACKET_HEADER_SIZE + adaptionFieldSize, b, 0, pesSegmentSize);

        return b;
    }

    /**
     * Extracts PES Header from Transport Packet.
     * 
     * @param tsFrame
     * 
     * @return
     */
    private byte[] getPesDataHeader(byte[] tsFrame)
    {
        byte[] b = new byte[20];
        System.arraycopy(tsFrame, 4, b, 0, 16);

        return b;
    }
}
