package tsStreamRipper.mpeg;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.ConnectionClosedException;

/**
 * Diese Klasse synchronisiert auf einen Transportstrom. Listener werden über eingelesene Pakete informiert.
 */
public class MPEG2TransportStreamReader
{
    private static final Logger log = Logger.getLogger(MPEG2TransportStreamReader.class.getName());

    /**
     * Size of a TransportPacket
     */
    public final static int TRANSPORT_PACKET_SIZE = 188;

    /**
     * Size of Header
     */
    public final static int TRANSPORT_PACKET_HEADER_SIZE = 4;

    /**
     * This is the next available TransportPacket.
     */
    private byte[] tsFrame = new byte[TRANSPORT_PACKET_SIZE];

    /**
     * This is s temporarly buffer for syncing to the TransportStream.
     */
    private byte[] syncFrame = new byte[2 * TRANSPORT_PACKET_SIZE];

    /**
     * Is syncFrame drained
     */
    private boolean syncFrameDrained = false;

    /**
     * Position to where the syncFrame is already drained.
     */
    private int drainPosition = 0;

    /**
     * Listener will be notified about received packets.
     */
    private java.util.List<ITransportPacketListener> listener = new ArrayList<ITransportPacketListener>();

    /**
     * TS-Stream
     */
    private InputStream is = null;

    /**
     * Is Stream synced
     */
    private boolean isSynced = false;

    /**
     * 
     * @param is
     *            Source InputStream. This Stream should be a MPEG2-TransportStream.
     * @param listener
     *            At least one listener should be here ...
     */
    public MPEG2TransportStreamReader(InputStream is, ITransportPacketListener listener)
    {
        this.is = is;
        addListener(listener);
    }

    /**
     * 
     * @return true : Buffer read<bR>
     *         false : Stream closed
     * @throws IOException
     */
    public boolean decodeNextFrame() throws IOException
    {
        // read next frame
        readNextFrame();

        //
        // inform listener with a copy of frame ...
        //
        byte[] frame = new byte[TRANSPORT_PACKET_SIZE];
        System.arraycopy(tsFrame, 0, frame, 0, TRANSPORT_PACKET_SIZE);

        for (ITransportPacketListener aListener : listener)
        {
            aListener.transportPackedReceived(frame);
        }

        return true;
    }

    /**
     * Reads next TransportPacket into buffer tsFrame
     * 
     * @return Bytes read.
     * 
     * @throws IOException
     */
    private int readNextFrame() throws IOException
    {
        // check if we are synced
        if (!isSynced)
        {
            if (log.isLoggable(Level.INFO))
                log.info("synchronisiere Transportstrom. Dabei kann es zu Warnmeldungen kommen.");
            sync();
            if (!isSynced)
            {
                log.severe("Transportstrom läßt sich nicht synchronisieren!");
                throw new ConnectionClosedException("cannot sync. Maybe this is not a TransportStream");
            }
        }
        // check, if we have data in syncFrame.
        else if (!syncFrameDrained)
        {
            System.arraycopy(syncFrame, 0, tsFrame, 0, 188);

            int bytesRead = readBytesIntoBuffer(is, tsFrame, drainPosition, TRANSPORT_PACKET_SIZE - drainPosition);
            if (bytesRead == -1)
            {
                log.severe("Stream closed");
                throw new ConnectionClosedException("stream closed while syncing frame");
            }
            syncFrameDrained = true;
        }
        else
        {
            // read data from InputStream
            int bytesRead = readBytesIntoBuffer(is, tsFrame);
            if (bytesRead == -1)
            {
                log.info("Transportsstrom geschlossen ...");
                throw new ConnectionClosedException("stream closed.");
            }
            if (bytesRead < 188)
            {
                log.warning("Buffer underrun. Stream is probably closing ...");
            }
        }

        // sync Frame exists ?
        if (!((tsFrame[0] & 0xff) == 0x47))
        {
            log.warning("Stream corrupt. SYNC lost. Trying to resync.");
            isSynced = false;
            readNextFrame();
        }

        return TRANSPORT_PACKET_SIZE;
    }

    /**
     * Sync to MPEG2TransportStream packets.
     * 
     * After this operation, the first synced frame is copied into the FrameBuffer tsFrame
     * 
     * @throws IOException
     */
    private boolean sync() throws IOException
    {
        isSynced = false;
        syncFrameDrained = false;

        readBytesIntoBuffer(is, syncFrame);
        HashSet<Integer> syncPoints = new HashSet<Integer>();
        for (int i = 0; i < syncFrame.length; i++)
        {
            if (syncFrame[i] == 0x47)
            {
                for (Integer sp : syncPoints)
                {
                    if (sp + 188 == i)
                    {
                        isSynced = true;

                        if (log.isLoggable(Level.FINER))
                            log.finer("TransportPacket is synced at stream position : " + sp);

                        // Copy first frame into tsFrame
                        System.arraycopy(syncFrame, sp, tsFrame, 0, 188);

                        // Copy remaining bytes to start of syncFrame
                        byte[] tmpFrame = new byte[188];
                        drainPosition = 188 - sp;
                        System.arraycopy(syncFrame, sp + 188, tmpFrame, 0, drainPosition);
                        System.arraycopy(tmpFrame, 0, syncFrame, 0, drainPosition);
                        return true;
                    }
                }
                syncPoints.add(i);
            }
        }

        return false;
    }

    /**
     * This method reads bytes from the InputStream until it is filled.
     * 
     * @param is
     *            The Stream
     * @param buffer
     *            Buffer to be filled.
     * @return
     * @throws IOException
     */
    private int readBytesIntoBuffer(InputStream is, byte[] buffer) throws IOException
    {
        return readBytesIntoBuffer(is, buffer, 0, buffer.length);
    }

    private int readBytesIntoBuffer(InputStream is, byte[] buffer, int pos, int length) throws IOException
    {
        int count = is.read(buffer, pos, length);
        if (count == -1)
            return -1;

        count += pos;

        while (count < buffer.length)
        {
            int br = is.read(buffer, count, buffer.length - count);
            if (br == -1)
            {
                log.warning("Last message before STREAM CLOSED : " + new String(buffer));
                throw new IOException("STREAM CLOSED AT FRAME POS : " + count);
            }
            count = count + br;
        }
        return count;
    }

    /**
     * Adds an TS-listener
     * 
     * @param listener
     */
    public void addListener(ITransportPacketListener listener)
    {
        this.listener.add(listener);
    }

    /**
     * Removes a TS-Listener
     * 
     * @param listener
     */
    public void removeListener(ITransportPacketListener listener)
    {
        this.listener.remove(listener);
    }
}
