﻿# -*- coding: UTF-8 -*-
from __future__ import absolute_import
from __future__ import print_function
from Plugins.Plugin import PluginDescriptor
from Tools.Downloader import downloadWithProgress
from Tools.Directories import fileExists, resolveFilename, SCOPE_PLUGINS
from twisted.web.client import getPage
from enigma import ePicLoad, eServiceReference, getDesktop, eTimer
from Screens.Screen import Screen
from Screens.ChoiceBox import ChoiceBox
from Screens.VirtualKeyBoard import VirtualKeyBoard
from Components.ActionMap import ActionMap
from Components.Pixmap import Pixmap
from Components.Label import Label
from Components.ScrollLabel import ScrollLabel
from Components.Button import Button
from Components.AVSwitch import AVSwitch
from Components.MenuList import MenuList
from Components.ProgressBar import ProgressBar
from Components.Sources.StaticText import StaticText
from Components.config import config, getConfigListEntry, ConfigSubsection, ConfigYesNo, ConfigText, ConfigSelection
from Components.ConfigList import ConfigListScreen
from Components.PluginComponent import plugins
import os, re, random, string, six
from six.moves.urllib.parse import quote_plus
from time import time, strftime
import json
from six.moves.html_parser import HTMLParser

version = "0.1-r10"
debug_output = False

try:
	import htmlentitydefs
except ImportError as ie:
	from html import entities as htmlentitydefs
	unichr = chr

sz_w = getDesktop(0).size().width()

config.plugins.imdb = ConfigSubsection()
config.plugins.imdb.showinplugins = ConfigYesNo(default = True)
config.plugins.imdb.showepisodeinfo = ConfigYesNo(default = True)
config.plugins.imdb.origskin = ConfigYesNo(default = True)
config.plugins.imdb.ignore_tags = ConfigText(visible_width = 50, fixed_size = False)
config.plugins.imdb.language = ConfigSelection(default = None, choices = [(None, _("Default")),("de-de", _("German")),("en-us", _("English")),("fr-fr", _("French")),("it-it", _("Italian")),("es-es", _("Spanish"))])
config.plugins.imdb.parsingmethod = ConfigSelection(default = "json", choices = [(None, _("Default")),("json", _("Alternativ"))])

imdb_headers = {}

useragents = [
	'Mozilla/5.0 (compatible; Konqueror/4.5; FreeBSD) KHTML/4.5.4 (like Gecko)',
	'Mozilla/5.0 (compatible; MSIE 10.0; Windows NT 6.1; WOW64; Trident/6.0)',
	'Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 7.1; Trident/5.0)',
	'Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/36.0.1985.67 Safari/537.36',
	'Mozilla/5.0 (Windows NT 6.1; WOW64; rv:35.0) Gecko/20120101 Firefox/35.0',
	'Mozilla/5.0 (Windows NT 6.3; rv:36.0) Gecko/20100101 Firefox/36.0',
	'Mozilla/5.0 (X11; Linux x86_64; rv:28.0) Gecko/20100101 Firefox/28.0',
	'Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:91.0) Gecko/20100101 Firefox/91.0',
	'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/92.0.4515.159 Safari/537.36'
	]

def getRandomUserAgent():
	return random.choice(useragents)

def quoteEventName(eventName):
	try:
		text = eventName.decode('utf8').replace(u'\x86', u'').replace(u'\x87', u'').encode('utf8')
	except:
		text = eventName
	return quote_plus(text, safe="+")

class HTMLFilter(HTMLParser):
	text = ''
	def handle_data(self, data):
		self.text += data

def html2text(html):
	filter = HTMLFilter()
	filter.feed(str(html))
	return filter.text

def getJsonValues(jsonData, jsonKey, default=None):
	try:
		for item in jsonKey:
			jsonData = jsonData[item]
		return jsonData
	except (KeyError, TypeError, IndexError):
		return default

class IMDB(Screen):
	if sz_w == 1920:
		skin = """
		<screen name="IMDBv2" position="center,110" size="1800,930" title="IMDb - Internet Movie Database">
		<eLabel backgroundColor="grey" position="10,80" size="1780,1" />
		<ePixmap pixmap="Default-FHD/skin_default/buttons/red.svg" position="10,5" size="300,70" />
		<ePixmap pixmap="Default-FHD/skin_default/buttons/green.svg" position="310,5" size="300,70" />
		<ePixmap pixmap="Default-FHD/skin_default/buttons/yellow.svg" position="610,5" size="300,70" />
		<ePixmap pixmap="Default-FHD/skin_default/buttons/blue.svg" position="910,5" size="300,70" />
		<widget backgroundColor="#9f1313" font="Regular;30" halign="center" name="key_red" position="10,5" foregroundColor="white" shadowColor="black" shadowOffset="-2,-2" size="300,70" transparent="1" valign="center" zPosition="1" />
		<widget backgroundColor="#1f771f" font="Regular;30" halign="center" name="key_green" position="310,5" foregroundColor="white" shadowColor="black" shadowOffset="-2,-2" size="300,70" transparent="1" valign="center" zPosition="1" />
		<widget backgroundColor="#a08500" font="Regular;30" halign="center" name="key_yellow" position="610,5" foregroundColor="white" shadowColor="black" shadowOffset="-2,-2" size="300,70" transparent="1" valign="center" zPosition="1" />
		<widget backgroundColor="#18188b" font="Regular;30" halign="center" name="key_blue" position="910,5" foregroundColor="white" shadowColor="black" shadowOffset="-2,-2" size="300,70" transparent="1" valign="center" zPosition="1" />
		<widget font="Regular;34" halign="right" position="1650,25" render="Label" size="120,40" source="global.CurrentTime">
			<convert type="ClockToText">Default</convert>
		</widget>
		<widget font="Regular;34" halign="right" position="1240,25" render="Label" size="400,40" source="global.CurrentTime" >
			<convert type="ClockToText">Date</convert>
		</widget>
		<widget font="Regular;38" name="titlelabel" position="20,90" size="1780,45" foregroundColor="yellow"/>
		<widget enableWrapAround="1" name="menu" position="20,150" scrollbarMode="showOnDemand" size="1760,720" />
		<widget font="Regular;30" name="extralabel" position="20,150" size="1760,730" />
		<widget font="Regular;30" halign="left" name="ratinglabel" position="340,150" size="1000,40" foregroundColor="yellow"/>
		<widget name="starsbg" pixmap="/usr/lib/enigma2/python/Plugins/Extensions/IMDb/starsbar_empty.svg" position="20,150" size="300,31" />
		<widget name="stars" pixmap="/usr/lib/enigma2/python/Plugins/Extensions/IMDb/starsbar_filled.svg" position="20,150" size="300,31" transparent="1" />
		<widget name="poster" position="20,200" size="300,449" />
		<widget font="Regular;30" name="detailslabel" position="340,210" size="770,400" />
		<widget font="Regular;30" name="castlabel" position="1130,210" size="650,400" />
		<widget font="Regular;30" name="storylinelabel" position="20,660" size="1760,210" />
		<widget name="statusbar" halign="left" position="20,885" size="1600,35" font="Regular;30" foregroundColor="#b3b3b9" />
		</screen>"""
	elif sz_w == 2560:
		skin = """
		<screen name="IMDBv2" position="center,160" size="2400,1220" title="IMDb - Internet Movie Database">
		<ePixmap pixmap="skin_default/buttons/red.png" position="20,10" size="400,80"/>
		<ePixmap pixmap="skin_default/buttons/green.png" position="420,10" size="400,80"/>
		<ePixmap pixmap="skin_default/buttons/yellow.png" position="820,10" size="400,80"/>
		<ePixmap pixmap="skin_default/buttons/blue.png" position="1220,10" size="400,80"/>
		<widget name="key_red" position="20,10" size="400,80" zPosition="1" font="Regular;40" halign="center" valign="center" backgroundColor="#9f1313" transparent="1" foregroundColor="white" shadowColor="black" shadowOffset="-4,-4"/>
		<widget name="key_green" position="420,10" size="400,80" zPosition="1" font="Regular;40" halign="center" valign="center" backgroundColor="#1f771f" transparent="1" foregroundColor="white" shadowColor="black" shadowOffset="-4,-4"/>
		<widget name="key_yellow" position="820,10" size="400,80" zPosition="1" font="Regular;40" halign="center" valign="center" backgroundColor="#a08500" transparent="1" foregroundColor="white" shadowColor="black" shadowOffset="-4,-4"/>
		<widget name="key_blue" position="1220,10" size="400,80" zPosition="1" font="Regular;40" halign="center" valign="center" backgroundColor="#18188b" transparent="1" foregroundColor="white" shadowColor="black" shadowOffset="-4,-4"/>
		<widget source="global.CurrentTime" render="Label" position="2240,24" size="120,50" font="Regular;44" halign="right">
			<convert type="ClockToText">Default</convert>
		</widget>
		<widget source="global.CurrentTime" render="Label" position="1620,24" size="600,50" font="Regular;44" halign="right">
			<convert type="ClockToText">Format:%A %d. %B</convert>
		</widget>
		<eLabel position="20,100" size="2360,2" backgroundColor="grey"/>
		<widget name="titlelabel" position="40,110" size="2360,60" valign="center" font="Regular;48" foregroundColor="yellow"/>
		<widget name="menu" position="40,180" size="2320,960" zPosition="3" scrollbarMode="showOnDemand" enableWrapAround="1" />
		<widget name="extralabel" position="40,180" size="2320,980" font="Regular;40" />
		<widget name="ratinglabel" position="450,184" size="1600,46" font="Regular;40" foregroundColor="yellow"/>
		<widget name="starsbg" pixmap="/usr/lib/enigma2/python/Plugins/Extensions/IMDb/starsbar_empty.svg" position="40,180" size="370,42"/>
		<widget name="stars" pixmap="/usr/lib/enigma2/python/Plugins/Extensions/IMDb/starsbar_filled.svg" position="40,180" size="370,42" transparent="1" />
		<widget name="poster" position="40,240" size="370,556"/>
		<widget name="detailslabel" position="450,260" size="900,540" font="Regular;40" />
		<widget name="castlabel" position="1390,260" size="970,560" font="Regular;40" />
		<widget name="storylinelabel" position="40,820" size="2320,340" font="Regular;40" />
		<widget name="statusbar" position="40,1160" size="2300,50" halign="left" font="Regular;40" foregroundColor="#b3b3b9" />
		</screen>"""
	else:
		skin = """
		<screen name="IMDBv2" position="center,80" size="1200,610" title="IMDb - Internet Movie Database">
		<ePixmap pixmap="skin_default/buttons/red.png" position="10,5" size="200,40"/>
		<ePixmap pixmap="skin_default/buttons/green.png" position="210,5" size="200,40"/>
		<ePixmap pixmap="skin_default/buttons/yellow.png" position="410,5" size="200,40"/>
		<ePixmap pixmap="skin_default/buttons/blue.png" position="610,5" size="200,40"/>
		<widget name="key_red" position="10,5" size="200,40" zPosition="1" font="Regular;20" halign="center" valign="center" backgroundColor="#9f1313" transparent="1" foregroundColor="white" shadowColor="black" shadowOffset="-2,-2"/>
		<widget name="key_green" position="210,5" size="200,40" zPosition="1" font="Regular;20" halign="center" valign="center" backgroundColor="#1f771f" transparent="1" foregroundColor="white" shadowColor="black" shadowOffset="-2,-2"/>
		<widget name="key_yellow" position="410,5" size="200,40" zPosition="1" font="Regular;20" halign="center" valign="center" backgroundColor="#a08500" transparent="1" foregroundColor="white" shadowColor="black" shadowOffset="-2,-2"/>
		<widget name="key_blue" position="610,5" size="200,40" zPosition="1" font="Regular;20" halign="center" valign="center" backgroundColor="#18188b" transparent="1" foregroundColor="white" shadowColor="black" shadowOffset="-2,-2"/>
		<widget source="global.CurrentTime" render="Label" position="1120,12" size="60,25" font="Regular;22" halign="right">
			<convert type="ClockToText">Default</convert>
		</widget>
		<widget source="global.CurrentTime" render="Label" position="810,12" size="300,25" font="Regular;22" halign="right">
			<convert type="ClockToText">Format:%A %d. %B</convert>
		</widget>
		<eLabel position="10,50" size="1180,1" backgroundColor="grey"/>
		<widget name="titlelabel" position="20,55" size="1180,30" valign="center" font="Regular;24" foregroundColor="yellow"/>
		<widget name="menu" position="20,90" size="1160,480" zPosition="3" scrollbarMode="showOnDemand" enableWrapAround="1" />
		<widget name="extralabel" position="20,90" size="1160,490" font="Regular;20" />
		<widget name="ratinglabel" position="225,92" size="800,23" font="Regular;20" foregroundColor="yellow"/>
		<widget name="starsbg" pixmap="/usr/lib/enigma2/python/Plugins/Extensions/IMDb/starsbar_empty.svg" position="20,90" size="185,21"/>
		<widget name="stars" pixmap="/usr/lib/enigma2/python/Plugins/Extensions/IMDb/starsbar_filled.svg" position="20,90" size="185,21" transparent="1" />
		<widget name="poster" position="20,120" size="185,278"/>
		<widget name="detailslabel" position="225,130" size="450,270" font="Regular;20" />
		<widget name="castlabel" position="695,130" size="485,280" font="Regular;20" />
		<widget name="storylinelabel" position="20,410" size="1160,170" font="Regular;20" />
		<widget name="statusbar" position="20,580" size="1150,25" halign="left" font="Regular;20" foregroundColor="#b3b3b9" />
		</screen>"""

	NBSP = unichr(htmlentitydefs.name2codepoint['nbsp']).encode("utf8")

	def __init__(self, session, eventName, callbackNeeded=False):
		Screen.__init__(self, session)
		if config.plugins.imdb.origskin.value:
			self.skinName = ''.join([random.choice(string.ascii_letters + string.digits) for n in xrange(32)])
		else:
			self.skinName = "IMDBv2"

		for tag in config.plugins.imdb.ignore_tags.getValue().split(','):
			eventName = eventName.replace(tag,'')

		self.eventName = eventName

		self.callbackNeeded = callbackNeeded
		self.callbackData = ""
		self.callbackGenre = ""
		self.manualsearch = False

		self.dictionary_init()

		self["poster"] = Pixmap()
		self.picload = ePicLoad()
		self.picload_conn = self.picload.PictureData.connect(self.paintPosterPixmapCB)

		self["stars"] = ProgressBar()
		self["starsbg"] = Pixmap()
		self["stars"].hide()
		self["starsbg"].hide()
		self.ratingstars = -1

		self.setTitle(_("IMDb - Internet Movie Database"))
		self["titlelabel"] = Label("")
		self["titlelcd"] = StaticText("")
		self["detailslabel"] = ScrollLabel("")
		self["castlabel"] = ScrollLabel("")
		self["storylinelabel"] = ScrollLabel("")
		self["extralabel"] = ScrollLabel("")
		self["statusbar"] = Label("")
		self["ratinglabel"] = Label("")
		self.resultlist = []
		self["menu"] = MenuList(self.resultlist)
		self["menu"].hide()

		self["key_red"] = Button(_("Search"))
		self["key_green"] = Button("")
		self["key_yellow"] = Button("")
		self["key_blue"] = Button("")

		# 0 = multiple query selection menu page
		# 1 = movie info page
		# 2 = extra infos page
		self.Page = 0

		self["actions"] = ActionMap(["OkCancelActions", "ColorActions", "IMDbActions", "InfobarActions", "MovieSelectionActions", "DirectionActions"],
		{
			"ok": self.showDetails,
			"cancel": self.exit,
			"down": self.pageDown,
			"up": self.pageUp,
			"right": self.keyRight,
			"left": self.keyLeft,
			"red_normal": self.openVirtualKeyBoard,
			"red_long": self.openVirtualKeyBoardWithLastSearch,
			"green": self.showMenu,
			"yellow": self.showDetails,
			"blue": self.showExtrasFirst,
			"contextMenu": self.contextMenuPressed,
			"showEventInfo": self.showDetails,
			"showMovies": self.getIMDB
		}, -1)

		self.getIMDB()

	def title_setText(self, txt):
		StaticText.setText(self["titlelcd"], txt)
		self["titlelabel"].setText(txt)

	def exit(self):
		if fileExists("/tmp/poster.jpg"):
			os.remove("/tmp/poster.jpg")
		if self.callbackNeeded:
			self.close([self.callbackData, self.callbackGenre])
		else:
			self.close()

	def dictionary_init(self):
		# self.generalinfomask = re.compile(
		# '<h1.*?hero-title-block__title" class=.*?> (?P<title>.*?)</h1>*'
		# '(?:.*?>(?P<g_episodes>Episodes)<span.*?>(?P<episodes>.*?)</span>)?'
		# '(?:.*?class="OriginalTitle__OriginalTitleText.*?>(?P<g_originaltitle>.*?):\s.*?(?P<originaltitle>.*?)</div)?'
		# '(?:.*?<label for="browse-episodes-season".*?>.*?(?P<seasons>\d+)\s(?P<g_seasons>seasons?)</label>)?'
		# '(?:.*?<span.*?>(?P<g_director>Directors?)(?:</span>|</a>).*?<div.*?<ul.*?>(?P<director>.*?)</ul>)?'
		# '(?:.*?<span.*?>(?P<g_creator>Creators?)(?:</span>|</a>).*?<div.*?<ul.*?>(?P<creator>.*?)</ul>)?'
		# '(?:.*?<span.*?>(?P<g_writer>Writers?)(?:</span>|</a>).*?<div.*?<ul.*?>(?P<writer>.*?)</ul>)?'
		# '(?:.*?<a.*?>(?P<g_premiere>Release date)</a>.*?<div.*?<ul.*?>(?P<premiere>.*?)</ul>)?'
		# '(?:.*?<span.*?>(?P<g_country>Countr.*?of origin)</span>.*?<div.*?<ul.*?>(?P<country>.*?)</ul>)?'
		# '(?:.*?<a.*?>(?P<g_alternativ>Also known as)</a>.*?<div.*?<ul.*?>(?P<alternativ>.*?)</ul>)?'
		# '(?:.*?techspec_runtime.*?>(?P<g_runtime>Runtime)</span>.*?<div.*?>(?P<runtime>.*?)</div)?', re.S)
		
		self.generalinfomask = re.compile(
		'(<h1.*?hero-title-block__title" class=.*?>|<h1.*?hero__pageTitle" class=.*?><span.*?>)(?P<title>.*?)</*'
		'(?:.*?>(?P<g_episodes>Episodes)</span><span.*?>(?P<episodes>.*?)</span>)?'
		'(?:.*?(<div.*?original-title.*?>|</h1><div class=.*?>)(?P<g_originaltitle>.*?):\s.*?(?P<originaltitle>.*?)</div*)?'
		'(?:.*?<button.*?>(?P<g_director>Director.?)(</button>|</a>).*?<div.*?<ul.*?>(?P<director>.*?)</ul>)?'
		'(?:.*?<button.*?>(?P<g_writer>Writer.?)(</button>|</a>)<div.*?<ul.*?>(?P<writer>.*?)</ul>)?'
		'(?:.*?<button.*?>(?P<g_creator>Creator.?)(</button>|</a>)<div.*?<ul.*?>(?P<creator>.*?)</ul>)?'
		'(?:.*?<label for="browse-episodes-season".*?>.*?(?P<seasons>\d+)\s(?P<g_seasons>seasons?)</label>)?'
		'(?:.*?<a.*?>(?P<g_premiere>Release date)</a>.*?<div.*?<ul.*?>(?P<premiere>.*?)</ul>)?'
		'(?:.*?<button.*?>(?P<g_country>Countr.*?of origin)</button>.*?<div.*?<a.*?>(?P<country>.*?)</a>)?'
		'(?:.*?<a.*?>(?P<g_alternativ>Also known as)</a>.*?<label.*?>(?P<alternativ>.*?)</label>)?'
		'(?:.*?techspec_runtime.*?><button.*?>(?P<g_runtime>Runtime)</button>.*?<div.*?>(?P<runtime>.*?)</div)?', re.S)
		
		self.certificatemask = re.compile('<a.class.*?parental.*?>(?P<cert>.*?)</a>', re.S)
		
		self.extrainfomask = re.compile(
		'(?:.*?<div.*?class="GenresAndPlot__TextContainerBreakpointXL.*?>(?P<outline>.+?)</div>)?'
		'(?:.*?<a.*?>(?P<g_tagline>Taglines?)(?:</span>|</a>).*?<div.*?<ul.*?<li.*?<span.*?>(?P<tagline>.*?)</span>)?'
		'(?:.*?<a.*?>(?P<g_cert>Certificate|Motion Picture Rating \(MPAA\))</a>.*?<div.*?<ul.*?<li.*?<span.*?>(?P<cert>.*?)</span>)?'
		'(?:.*?<a.*?>(?P<g_trivia>Trivia)</a><div.*?<div.*?<div.*?<div.*?>(?P<trivia>.+?)</div>)?'
		'(?:.*?<a.*?>(?P<g_goofs>Goofs)</a><div.*?<div.*?<div.*?<div.*?>(?P<goofs>.+?)</div>)?'
		'(?:.*?<a.*?>(?P<g_quotes>Quotes)</a><div.*?<div.*?<div.*?<div.*?>(?P<quotes>.+?)</div>)?'
		'(?:.*?<a.*?>(?P<g_crazycredits>Crazy credits)</a><div.*?<div.*?<div.*?<div.*?>(?P<crazycredits>.+?)</div>)?'
		'(?:.*?<a.*?>(?P<g_alternateversions>Alternate versions)</a><div.*?<div.*?<div.*?<div.*?>(?P<alternateversions>.+?)</div>)?'
		'(?:.*?<a.*?>(?P<g_connections>Connections)</a><div.*?<div.*?<div.*?<a.*?>(?P<connections>.+?)</a>)?'
		'(?:.*?<a.*?>(?P<g_soundtracks>Soundtracks)</a><div.*?<div.*?<div.*?<div.*?>(?P<soundtracks>.+?)</div>)?'
		'(?:.*?<h3.*?>(?P<g_comments>User review)s.*?</h3>.*?<span.*?review-summary.*?>(?P<commenttitle>.*?)</span></div><div.*?<div.*?<div.*?>(?P<comment>.+?)</div>.*?<div.*?reviews-author.*?>.*?<ul.*?<li.*?>(?P<commenter>.+?)</li>)?'
		'(?:.*?<span.*?>(?P<g_language>Languages?)</span>.*?<div.*?<ul.*?>(?P<language>.*?)</ul>)?'
		'(?:.*?<a.*?>(?P<g_locations>Filming locations?)</a>.*?<div.*?<ul.*?>(?P<locations>.*?)</ul>)?'
		'(?:.*?<a.*?>(?P<g_company>Production compan.*?)</a>.*?<div.*?<ul.*?>(?P<company>.*?)</ul>)?'
		'(?:.*?<button.*?>(?P<g_color>Color)</button>.*?<div.*?<a.*?>(?P<color>.*?)</a>)?'
		'(?:.*?<button.*?>(?P<g_sound>Sound mix)</button>.*?<div.*?<a.*?>(?P<sound>.*?)</a>)?'
		'(?:.*?<button.*?>(?P<g_aspect>Aspect ratio)</button>.*?<div.*?<ul.*?<li.*?<label.*?>(?P<aspect>.*?)</label>)?', re.S)

		#self.awardsmask = re.compile('(?:.*?<li.*?data-testid="award_information".*?><a.*?>(?P<awards>.+?)</span></li>)?', re.S)
		
		self.keywordsmask = re.compile('data-testid="storyline-plot-keywords">(?P<keywords>.*?)</div>', re.S)
		
		self.boxofficemask = re.compile('<h3.*?>(?P<g_boxoffice>Box office)</h3.*?<div.*?list-item__label">Budget</span>.*?list-content-item">(?P<boxofficebudget>.*?)</span>.*?list-content-item">(?P<boxofficegrossuscanada>.*?)</span>.*?list-content-item">(?P<boxofficeopening>.*?)</span>.*?list-content-item">(?P<boxofficeopeningdate>.*?)</span>.*?list-content-item">(?P<boxofficegrossworld>.*?)</span>', re.S)
		
		#self.genreblockmask = re.compile('<li.*?storyline-genres.*?><span.*?>(?P<g_genres>Genres?)</span>.*?<div.*?><ul.*?>(?P<genres>.*?)</ul>', re.S)
		self.genreblockmask = re.compile('<div.*?data-testid="genres"><div.*?></div><div.*?>(?P<genres>.*?)</div>', re.S)
		
		self.ratingmask = re.compile('<span.*?aggregate-rating__score.*?><span.*?>(?P<rating>.*?)</span>.*?<span.*?class=.*?class=.*?>(?P<ratingcount>.*?)</div', re.S)
		
		#self.castmask = re.compile('title-cast-item__actor.*?>(?P<actor>.*?)</a>(?:<div.*?<ul.*?>(?P<character>.*?)</span.*?</ul></div>)?(?:<button.*?><div.*?><span.*?>(?P<episodes>.*?)</span></div></button>)?', re.S)
		
		self.castmask = re.compile('title-cast-item__actor.*?>(?P<actor>.*?)</a>(?:<div.*?<ul.*?<span.*?>(?P<character>.*?)</span.*?</ul></div>)?(?:<button.*?><span.*?><span.*?>(?P<episodes>.*?)</span></span></button>)?', re.S)
		
		self.postermask = re.compile('<div.*?ipc-media--poster.*?<img.*?ipc-image.*?src="(http.*?)"', re.S)
		
		self.storylinemask = re.compile('<span.*?role="presentation".*?data-testid="plot-xl".*?>(?P<storyline>.*?)</span>.*?</div>.*?data-testid="storyline.*?<span>(?P<g_storyline>Storyline)</span>', re.S)
		
		self.htmltags = re.compile('<.*?>', re.S)
		
		self.allhtmltags = re.compile('<.*>', re.S)

	def resetLabels(self):
		self["detailslabel"].setText("")
		self["ratinglabel"].setText("")
		self["castlabel"].setText("")
		self["storylinelabel"].setText("")
		if not len(self.resultlist) >= 1:
			self["key_green"].setText("")
		self["key_yellow"].setText("")
		self["key_blue"].setText("")
		self.title_setText("")
		self["extralabel"].setText("")
		self.ratingstars = -1
		self["stars"].setValue(self.ratingstars)

	def pageUp(self):
		if self.Page == 0:
			self["menu"].instance.moveSelection(self["menu"].instance.moveUp)
		if self.Page == 1:
			self["castlabel"].pageUp()
			self["detailslabel"].pageUp()
			self["storylinelabel"].pageUp()
		if self.Page == 2:
			self["extralabel"].pageUp()

	def pageDown(self):
		if self.Page == 0:
			self["menu"].instance.moveSelection(self["menu"].instance.moveDown)
		if self.Page == 1:
			self["castlabel"].pageDown()
			self["detailslabel"].pageDown()
			self["storylinelabel"].pageDown()
		if self.Page == 2:
			self["extralabel"].pageDown()

	def keyLeft(self):
		if self.Page == 0:
			self["menu"].pageUp()
		if self.Page == 1:
			self["castlabel"].pageUp()
			self["detailslabel"].pageUp()
			self["storylinelabel"].pageUp()
		if self.Page == 2:
			self["extralabel"].pageUp()

	def keyRight(self):
		if self.Page == 0:
			self["menu"].pageDown()
		if self.Page == 1:
			self["castlabel"].pageDown()
			self["detailslabel"].pageDown()
			self["storylinelabel"].pageDown()
		if self.Page == 2:
			self["extralabel"].pageDown()

	def showMenu(self):
		if ( self.Page is 1 or self.Page is 2 ) and self.resultlist:
			self["menu"].show()
			self["stars"].hide()
			self["starsbg"].hide()
			self["ratinglabel"].hide()
			self["castlabel"].hide()
			self["storylinelabel"].hide()
			self["poster"].hide()
			self["extralabel"].hide()
			self["detailslabel"].hide()
			self.title_setText(_("Please select the matching entry"))
			self["key_blue"].setText("")
			self["key_green"].setText(_("Title Menu"))
			self["key_yellow"].setText("")
			self["statusbar"].setText("")
			self.Page = 0

	def showDetails(self):
		self["ratinglabel"].show()
		self["castlabel"].show()
		self["detailslabel"].show()
		self["storylinelabel"].show()
		self["menu"].hide()

		if self.resultlist and self.Page == 0:
			link = self["menu"].getCurrent()[1]
			title = self["menu"].getCurrent()[0]
			if config.plugins.imdb.parsingmethod.value == "json" and config.plugins.imdb.language.value == "de-de":
				imdb_headers = {'Accept-Language': config.plugins.imdb.language.value}
			else:
				imdb_headers = {}
			self["statusbar"].setText(_("Loading ...") + " IMDb: %s" % (title))
			fetchurl = "https://www.imdb.com/title/" + link + "/"
			print("[IMDB] showDetails() downloading query " + fetchurl)
			getPage(fetchurl, agent=getRandomUserAgent(), headers=imdb_headers).addCallback(self.IMDBquery2).addErrback(self.http_failed)
			self.resetLabels()
			self.Page = 1

		if self.Page == 2:
			self["extralabel"].hide()
			self["poster"].show()
			if self.ratingstars > 0:
				self["stars"].show()
				self["stars"].setValue(self.ratingstars)
			self["starsbg"].show()
			self.Page = 1

	def showExtrasFirst(self):
		if self["key_blue"].getText():
			self.startTimer = eTimer()
			if config.plugins.imdb.parsingmethod.value == "json":
				self.startTimer_conn = self.startTimer.timeout.connect(self.showExtras_json)
			else:
				self.startTimer_conn = self.startTimer.timeout.connect(self.showExtras_html)
			if not self["extralabel"].getText():
				self["statusbar"].setText(_("IMDb extra infos parsing..."))
			self.startTimer.start(50,True)

	def showExtras_json(self):
		if self.Page == 1:
			if not self["extralabel"].getText():
			
				aboveTheFoldData = self.jsonData['aboveTheFoldData']
				mainColumnData = self.jsonData['mainColumnData']
				
				Extratext = ''
				extrainfos1 = {}
				
				#_("Tagline:"), _("Taglines:"), _("Motion Picture Rating (MPAA):"), _("Language:"), _("Filming location:"), _("Production company:") # translate tags
				
				# search awards
				awards_total_wins = getJsonValues(mainColumnData, ['wins','total'],0)
				awards_total_nominations = getJsonValues(mainColumnData, ['nominations','total'],0)
				awards_prestigious_wins = getJsonValues(mainColumnData, ['prestigiousAwardSummary','wins'],0)
				awards_prestigious_nominations = getJsonValues(mainColumnData, ['prestigiousAwardSummary','nominations'],0)
				
				if awards_total_wins or awards_total_nominations or awards_prestigious_wins or awards_prestigious_nominations:
					extrainfos1["awards_text"] = _("Awards:")
					awards_prestigious_text = getJsonValues(mainColumnData, ['prestigiousAwardSummary','award', 'text'],"")
					translationData = getJsonValues(self.jsonData, ['translationContext','i18n','translations','resources'],[])
					translation_text = getJsonValues(translationData, ['feature_awards_winsAndNominations'],"")
					extrainfos1["awards"] = self.getAwardsText(awards_total_wins, awards_total_nominations,  awards_prestigious_wins, awards_prestigious_nominations, awards_prestigious_text, translation_text)
				
				# search boxoffice data
				def setBoxofficeData(path, desc):
					value = int(getJsonValues(mainColumnData, path + ["amount"],0))
					if value:
						currency = str(getJsonValues(mainColumnData, path + ["currency"],""))
						if "boxoffice" in extrainfos1.keys() and extrainfos1["boxoffice"]:
							extrainfos1["boxoffice"] += ", "
						extrainfos1["boxoffice"] += desc + " " + '{0:n}'.format(value) + " " + currency
				
				extrainfos1["boxoffice"] = ""
				extrainfos1["boxoffice_text"] = _("Box office:")
				setBoxofficeData(["productionBudget","budget"], _("Budget"))
				setBoxofficeData(["worldwideGross","total"], _("Gross worldwide"))
				setBoxofficeData(["lifetimeGross","total"], _("Gross domestic"))
				setBoxofficeData(["openingWeekendGross","gross","total"], _("Gross domestic opening weekend"))
				
				extrainfos1["outline"] = getJsonValues(aboveTheFoldData, ['plot', 'plotText', 'plainText'],"")
				extrainfos1["outline_text"] = _("Plot outline:")
				
				extrainfos1["keywords"] = ", ".join(getJsonValues(keyword, ('node', 'text')) for keyword in getJsonValues(aboveTheFoldData, ['keywords', 'edges'],[]))
				extrainfos1["keywords_text"] = _("Keywords:")
				
				extrainfos1["locations"] = getJsonValues(mainColumnData,["filmingLocations","edges",0,"node","text"],"")
				extrainfos1["locations_text"] = _("Filming locations:")
				
				extrainfos1["company"] = ", ".join(getJsonValues(node,["node","company", "companyText", "text"],"") for node in getJsonValues(mainColumnData,["production","edges"],[]))
				extrainfos1["company_text"] = _("Production companies:")
				
				extrainfos1["trivia"] = html2text(getJsonValues(mainColumnData,['trivia', 'edges', 0, 'node', 'text', 'plaidHtml'],""))
				extrainfos1["trivia_text"] = _("Trivia:")
				
				extrainfos1["goofs"] = html2text(getJsonValues(mainColumnData,['goofs', 'edges', 0, 'node', 'text', 'plaidHtml'],""))
				extrainfos1["goofs_text"] = _("Goofs:")
				
				extrainfos1["crazycredits"] = html2text(getJsonValues(mainColumnData,['crazyCredits', 'edges', 0, 'node', 'text', 'plaidHtml'],""))
				extrainfos1["crazycredits_text"] = _("Crazy credits:")
				
				extrainfos1["alternateversions"] = html2text(getJsonValues(mainColumnData,['alternateVersions', 'edges', 0, 'node', 'text', 'plaidHtml'],""))
				extrainfos1["alternateversions_text"] = _("Alternate versions:")
				
				extrainfos1["color"] = getJsonValues(mainColumnData,['technicalSpecifications', 'colorations', 'items', 0, 'text'],"")
				extrainfos1["color_text"] = _("Color:")
				
				extrainfos1["sound"] = ", ".join(getJsonValues(mix,["text"],"") for mix in getJsonValues(mainColumnData,["technicalSpecifications","soundMixes","items"],[]))
				extrainfos1["sound_text"] = _("Sound mix:")
				
				extrainfos1["aspect"] = getJsonValues(mainColumnData,['technicalSpecifications', 'aspectRatios', 'items', 0, 'aspectRatio'],"")
				extrainfos1["aspect_text"] = _("Aspect ratio:")
				
				extrainfos1["language"] = ", ".join(getJsonValues(lang, ['text']) for lang in getJsonValues(mainColumnData, ['spokenLanguages', 'spokenLanguages'],[]))
				extrainfos1["language_text"] = _("Languages:")
				
				extrainfos1["commenttitle"] = str(getJsonValues(mainColumnData,['featuredReviews', 'edges', 0, 'node', 'summary', 'originalText'],""))
				extrainfos1["comment"] = str(html2text(getJsonValues(mainColumnData,['featuredReviews', 'edges', 0, 'node', 'text', 'originalText', 'plaidHtml'],"")))
				extrainfos1["commenter"] = str(getJsonValues(mainColumnData,['featuredReviews', 'edges', 0, 'node', 'author', 'nickName'],""))
				extrainfos1["comment_text"] = _("User reviews:")
				
				extrainfos1["soundtracks"] = getJsonValues(mainColumnData,['soundtrack','edges',0,'node','text'],"")
				if extrainfos1["soundtracks"]:
					soundtracks = getJsonValues(mainColumnData, ['soundtrack','edges',0,'node','comments'],[])
					if soundtracks is None: soundtracks = []
					comments = "\n".join(html2text(getJsonValues(comment, ['plaidHtml'])) for comment in soundtracks)
					if comments:
						extrainfos1["soundtracks"] += "\n" + comments
				extrainfos1["soundtracks_text"] = _("Soundtracks:")
				
				extrainfos1["connections"] = getJsonValues(mainColumnData,['connections','edges',0,'node','associatedTitle', 'titleText', 'text'],"")
				value = getJsonValues(mainColumnData,['connections','edges',0,'node','category', 'text'],"")
				if value:
					extrainfos1["connections"] = "%s " % value + extrainfos1["connections"]
				value = getJsonValues(mainColumnData,['connections','edges',0,'node','associatedTitle', 'series', 'series', 'titleText', 'text'],"")
				if value:
					extrainfos1["connections"] = "%s - " % value + extrainfos1["connections"]
				value = getJsonValues(mainColumnData,['connections','edges',0,'node','associatedTitle', 'releaseYear', 'year'],"")
				if value:
					extrainfos1["connections"] += " (%s)" % value
				extrainfos1["connections_text"] = _("Connections:")
				
				extrainfos1["tagline_text"] = _("Tagline:")
				extrainfos1["taglines_text"] = _("Taglines:")
				
				def getQuotes(lines):
					quotes = []
					for line in lines:
						stageDirection = getJsonValues(line, ['stageDirection'],"")
						character = getJsonValues(line, ['characters', 0, 'character'],"")
						txt = getJsonValues(line, ['text'],"")
						print("=== txt", txt, line)
						line_txt = ""
						if character:
							line_txt += character + ": "
							if stageDirection:
								line_txt += "[%s] " % stageDirection
							if txt:
								line_txt += "%s" % txt
						elif stageDirection:
							line_txt += "[%s]" % stageDirection
						if line_txt:
							quotes.append(line_txt)
					return quotes
				
				extrainfos1["quotes"] = "\n".join(getQuotes(getJsonValues(mainColumnData,['quotes','edges',0,'node','lines'],[]))).strip("\n")
				extrainfos1["quotes_text"] = _("Quotes:")
				
				if debug_output:
					from pprint import pprint
					pprint(extrainfos1)
				
				if extrainfos1:
					for category in ("awards", "outline", "keywords", "language", "color", "aspect", "sound", "locations", "company", "boxoffice", "trivia", "goofs", "quotes", "crazycredits", "alternateversions", "connections", "soundtracks", "comment"):
						try:
							if extrainfos1[category]:
								txt = ""
								sep = "\n" if category in ("outline", "boxoffice", "trivia", "goofs", "quotes", "connections", "crazycredits", "alternateversions", "soundtracks", "comment") else " "
								if category == "comment":
									if extrainfos1["commenttitle"]:
										txt = extrainfos1["commenttitle"] + " [" + extrainfos1["commenter"] + "]\n" + extrainfos1["comment"] + "\n"
								else:
									txt = str(extrainfos1[category])
								Extratext += "\n" + extrainfos1[category + "_text"] + sep + txt + "\n"
						except (KeyError, TypeError, IndexError):
							print("[IMDb] error on key", category)
				
				if Extratext:
					self["extralabel"].setText(Extratext.strip('\n'))
				else:
					self["extralabel"].setText("\nno extra Info found")
				self["statusbar"].setText(_("IMDb Details parsed"))
			
			self["extralabel"].show()
			self["detailslabel"].hide()
			self["castlabel"].hide()
			self["storylinelabel"].hide()
			self["poster"].hide()
			self["stars"].hide()
			self["starsbg"].hide()
			self["ratinglabel"].hide()
			self.Page = 2

	def getAwardsText(self, awards_total_wins, awards_total_nominations, awards_prestigious_wins, awards_prestigious_nominations, awards_prestigious_text, translation_text):
		
		# set default awards text
		won_one_txt = "won"
		won_other_txt = "wons"
		nom_one_txt = "nomination"
		nom_other_txt = "nominations"
		
		try:
			pos1 = translation_text.find("{numOfWins")
			if pos1 != -1:
				pos2 = translation_text.find("one {1 ", pos1)
				pos3 = translation_text.find("}", pos2)
				won_one_txt = translation_text[pos2+7:pos3]
				pos2 = translation_text.find("other {{", pos1)
				pos3 = translation_text.find("}}", pos2)
				won_other_txt = translation_text[pos2+19:pos3]
			pos1 = translation_text.find("{numOfNoms")
			if pos1 != -1:
				pos2 = translation_text.find("one {1 ", pos1)
				pos3 = translation_text.find("}", pos2)
				nom_one_txt = translation_text[pos2+7:pos3]
				pos2 = translation_text.find("other {{", pos1)
				pos3 = translation_text.find("}}", pos2)
				nom_other_txt = translation_text[pos2+19:pos3]
			
			def getWinText(value):
				if value == 1 or value == "1":
					return "%s %s" % (value, won_one_txt)
				else:
					return "%s %s" % (value, won_other_txt)
			def getNomText(value):
				if value == 1 or value == "1":
					return "%s %s" % (value, nom_one_txt)
				else:
					return "%s %s" % (value, nom_other_txt)
			
			awards_txt = ""
			if awards_prestigious_wins or awards_prestigious_nominations:
				awards_txt += "\n"
				if awards_prestigious_text:
					awards_txt += "%s - " % awards_prestigious_text
				if awards_prestigious_wins:
					awards_txt += getWinText(awards_prestigious_wins)
				if awards_prestigious_nominations:
					if awards_prestigious_wins:
						awards_txt += ", "
					awards_txt += getNomText(awards_prestigious_nominations)
			if awards_total_wins or awards_total_nominations:
				awards_txt += "\n" + _("overall") + " - "
				if awards_total_wins:
					awards_txt += getWinText(awards_total_wins)
				if awards_total_nominations:
					if awards_total_wins:
						awards_txt += ", "
					awards_txt += getNomText(awards_total_nominations)
		
		except:
			import traceback, sys
			traceback.print_exc()
			awards_txt = ""
		
		return awards_txt
	
	def showExtras_html(self):
		if self.Page == 1:
			
			if not self["extralabel"].getText():
				
				Extratext = ''
				
				print('###############8')
				extrainfos = self.extrainfomask.search(self.inhtml)
				
				if extrainfos:
					_("Tagline:"), _("Taglines:") # translate tags
					for category in ("outline", "tagline"):
						try:
							if extrainfos.group(category):
								sep = "\n" if category in ("outline") else " "
								if category == "outline":
									if "Add a plot" in extrainfos.group(category) or extrainfos.group(category) == "</span></p>":
										continue
									Extratext += "\n" + _("Plot outline:")
								else:
									Extratext += "\n" + _(extrainfos.group('g_' + category)+":")
								txt = ', '.join(re.split('\|+', self.htmltags.sub('|', extrainfos.group(category).replace("\n", ' ').replace("<br>", '\n').replace("<br/>", '\n').replace('<br />', '\n')).strip('|').replace(' |' + self.NBSP, '').replace(self.NBSP, ' ')))
								Extratext += sep + txt + "\n"
						except IndexError:
							pass
				
				print('###############9 - keywords')
				keywordsresult = self.keywordsmask.search(self.inhtml)
				if keywordsresult:
					keywords = keywordsresult.group('keywords').replace(' ', '_')
					keywords = ', '.join(self.htmltags.sub(' ', keywords).split())
					keywords = re.sub(', \d+ more', '', keywords.replace('_', ' '))
					Extratext += "\n" + _("Keywords:") + " " + keywords + "\n"

				print('###############9a - awards')
				awardsmask = re.compile('<li.*?award_information">.*?<a.class="ipc.*?>(?P<awards1>.*?)</a>.*?(<label.*?>|<span.*?>)(?P<awards2>.*?)(</label>|</span>)',re.S)
				awardsresult = awardsmask.search(self.inhtml)
				if awardsresult:
					print('###############9b')
					awards1 = awardsresult.group('awards1')
					awards2 = awardsresult.group('awards2')
					if awards1 or awards2:
						Extratext += "\n" + _("Awards:") + "\n" + awards1 + " - " + awards2 + "\n"
				
				print('###############10 - boxoffice')
				boxofficeresult = self.boxofficemask.search(self.inhtml)
				if boxofficeresult:
					boxoffice = "Budget " + boxofficeresult.group('boxofficebudget') + "\n" + "Opening weekend US & Canada " + boxofficeresult.group('boxofficeopening') + " (" + boxofficeresult.group('boxofficeopeningdate') + ")" + "\n" + "Gross US & Canada " + boxofficeresult.group('boxofficegrossuscanada') + "\n" "Gross worldwide " + boxofficeresult.group('boxofficegrossworld')
					Extratext += "\n" + _("Box office:") + "\n" + boxoffice + "\n"
				
				if extrainfos:
					_("Certificate:"), _("Motion Picture Rating (MPAA):"), _("Language:"), _("Languages:"), _("Color:"), _("Aspect ratio:"), _("Sound mix:"), _("Filming location:"), _("Filming locations:"), _("Production company:"), _("Production companies:"), _("Trivia:"), _("Goofs:"), _("Quotes:"), _("Crazy credits:"), _("Alternate versions:"), _("Connections:"), _("Soundtracks:"), _("User reviews:") # translate tags
					for category in ("cert", "language", "color", "aspect", "sound", "locations", "company", "trivia", "goofs", "quotes", "crazycredits", "alternateversions", "connections", "soundtracks"):
						try:
							if extrainfos.group(category):
								sep = "\n" if category in ("trivia", "goofs", "quotes", "connections", "crazycredits", "alternateversions", "soundtracks") else " "
								Extratext += "\n" + _(extrainfos.group('g_' + category)+":")
								if category in ("trivia", "goofs", "quotes", "connections", "crazycredits", "alternateversions", "soundtracks"):
									txt = extrainfos.group(category).replace("<li>", '• ').replace("</li>", '\n').replace('Read all</a>', '').replace("</a>", '').replace("<br>", '\n').replace("<br/>", '\n').replace('<br />', '\n').replace("</p><p>", '\n')
									txt = self.htmltags.sub('', txt).strip('\n')
								elif category in ("aspect", "sound"):
									txt = extrainfos.group(category).replace(' : ', ':').replace('</a><span class="ipc-metadata-list-item__list-content-item--subText">', ' ').replace('</li>', ', ')
									txt = self.htmltags.sub('', txt).strip(', ').strip()
								else:
									txt = ', '.join(re.split('\|+', self.htmltags.sub('|', extrainfos.group(category).replace("\n", ' ').replace("<br>", '\n').replace("<br/>", '\n').replace('<br />', '\n')).strip('|').replace(' |' + self.NBSP, '').replace(self.NBSP, ' ')))
								Extratext += sep + txt + "\n"
						except IndexError:
							pass
					try:
						if extrainfos.group("g_comments"):
							Extratext += "\n" + _(extrainfos.group("g_comments")+":") + "\n" + self.htmltags.sub('', extrainfos.group("commenttitle")) + " [" + ' '.join(self.htmltags.sub('', extrainfos.group("commenter")).split()) + "]\n" + self.htmltags.sub('', extrainfos.group("comment").replace("<br>", '\n').replace("<br/>", '\n').replace("<br />", '\n')) + "\n"
					except IndexError:
						pass
				
				if Extratext:
					self["extralabel"].setText(Extratext.strip('\n'))
				else:
					self["extralabel"].setText("\nno extra Info found")
				self["statusbar"].setText(_("IMDb Details parsed"))
			
			self["extralabel"].show()
			self["detailslabel"].hide()
			self["castlabel"].hide()
			self["storylinelabel"].hide()
			self["poster"].hide()
			self["stars"].hide()
			self["starsbg"].hide()
			self["ratinglabel"].hide()
			self.Page = 2

	def contextMenuPressed(self):
		list = [(_("Setup"), self.setup),]
		if fileExists(resolveFilename(SCOPE_PLUGINS, "Extensions/YTTrailer/plugin.py")):
			list.extend((
				(_("Play Trailer"), self.openYttrailer),
				(_("Search Trailer"), self.searchYttrailer),
			))
			self.session.openWithCallback(self.menuCallback, ChoiceBox, title=_("IMDb Menu"), list = list,)
		else:
			self.setup()

	def menuCallback(self, ret = None):
		ret and ret[1]()

	def openYttrailer(self):
		try:
			from Plugins.Extensions.YTTrailer.plugin import YTTrailer, baseEPGSelection__init__
		except ImportError as ie:
			pass
		if baseEPGSelection__init__ is None:
			return

		ytTrailer = YTTrailer(self.session)
		ytTrailer.showTrailer(self.eventName)

	def searchYttrailer(self):
		try:
			from Plugins.Extensions.YTTrailer.plugin import YTTrailerList, baseEPGSelection__init__
		except ImportError as ie:
			pass
		if baseEPGSelection__init__ is None:
			return

		self.session.open(YTTrailerList, self.eventName)

	def openVirtualKeyBoard(self):
		if self.manualsearch:
			text = self.eventName
		else:
			text = ''
			if debug_output: text = 'in the blood'
		self.session.openWithCallback(self.gotSearchString, VirtualKeyBoard, title = _("Enter text to search for"), text = text)

	def openVirtualKeyBoardWithLastSearch(self):
		text = self.eventName
		self.session.openWithCallback(self.gotSearchString, VirtualKeyBoard, title = _("Enter text to search for"), text = text)

	def gotSearchString(self, ret = None):
		if ret:
			self.eventName = ret
			self.Page = 0
			self.resultlist = []
			self["menu"].hide()
			self["ratinglabel"].show()
			self["castlabel"].show()
			self["storylinelabel"].show()
			self["detailslabel"].show()
			self["poster"].hide()
			self["stars"].hide()
			self["starsbg"].hide()
			self.manualsearch = True
			self.getIMDB()

	def hideAllFields(self):
		self["extralabel"].hide()
		self["detailslabel"].hide()
		self["castlabel"].hide()
		self["storylinelabel"].hide()
		self["poster"].hide()
		self["ratinglabel"].hide()
		self["menu"].hide()
		self["stars"].hide()
		self["starsbg"].hide()
		self["poster"].hide()

	def getIMDB(self):
		global imdb_headers
		self.hideAllFields()
		self.resultlist = []
		self["menu"].l.setList(self.resultlist)
		self.Page = 0
		
		imdb_headers = {}
		
		self.resetLabels()
		if not self.eventName:
			s = self.session.nav.getCurrentService()
			info = s and s.info()
			event = info and info.getEvent(0) # 0 = now, 1 = next
			if event:
				self.eventName = event.getEventName()
			else:
				self.eventName = self.session.nav.getCurrentlyPlayingServiceReference().toString()
				self.eventName = self.eventName.split('/')
				self.eventName = self.eventName[-1]
				self.eventName = self.eventName.replace('.',' ')
				self.eventName = self.eventName.split('-')
				self.eventName = self.eventName[0]
				if self.eventName.endswith(' '):
					self.eventName = self.eventName[:-1]
		if self.eventName:
			self["statusbar"].setText(_("Query IMDb: %s") % (self.eventName))
			fetchurl = "https://www.imdb.com/find/?ref_=nv_sr_fn&q=" + quoteEventName(self.eventName) + "&s=all"
			print("[IMDB] getIMDB() Downloading Query " + fetchurl)
			getPage(fetchurl, agent=getRandomUserAgent(), headers=imdb_headers).addCallback(self.IMDBquery).addErrback(self.http_failed)
		else:
			self["statusbar"].setText(_("Couldn't get Eventname"))

	def html2utf8(self, in_html):
		utf8 = ('charSet="utf-8"' in in_html or 'charset="utf-8"' in in_html or 'charSet=utf-8' in in_html or 'charset=utf-8' in in_html)
		start = in_html.find('<nav id="imdbHeader"')
		if start == -1:
			start = 0
		end = in_html.find('<section data-testid="contribution"')
		if end == -1:
			end = len(in_html)
		in_html = in_html[start:end]
		in_html = re.sub(r'(?s)<(script|style|svg).*?</\1>', '', in_html)
		entitydict = {}
		entities = re.finditer(r'&(?:([A-Za-z0-9]+)|#x([0-9A-Fa-f]+)|#([0-9]+));', in_html)
		for x in entities:
			key = x.group(0)
			if key not in entitydict:
				if x.group(1):
					if x.group(1) in htmlentitydefs.name2codepoint:
						entitydict[key] = htmlentitydefs.name2codepoint[x.group(1)]
				elif x.group(2):
					entitydict[key] = str(int(x.group(2), 16))
				else:
					entitydict[key] = x.group(3)
		if utf8:
			for key, codepoint in six.iteritems(entitydict):
				cp = six.unichr(int(codepoint))
				if six.PY2:
					cp = cp.encode('utf8')
				in_html = in_html.replace(key, cp)
			self.inhtml = in_html
		else:
			for key, codepoint in six.iteritems(entitydict):
				cp = six.unichr(int(codepoint))
				if six.PY2:
					cp = cp.encode('latin-1', 'ignore')
				in_html = in_html.replace(key, cp)
			if six.PY2:
				self.inhtml = in_html.decode('latin-1').encode('utf8')

	def IMDBquery(self, data, loadTitle=False):
		self["statusbar"].setText(_("IMDb Download completed"))
		self.data = data
		self.html2utf8(data)
		if debug_output:
			logfile = open("/media/hdd/imdb.html", "w")
			logfile.write(self.inhtml)
			logfile.close()
			logfile = open("/media/hdd/data.txt", "w")
			logfile.write(self.data)
			logfile.close()
		print('############### IMDBquery, loadTitle: ' + str(loadTitle))
		if config.plugins.imdb.parsingmethod.value == "json":
			if loadTitle:
				self.Page = 2
				self.showDetails()
				print('############### IMDBquery load IMDBparse')
				self.IMDBparse_json()
				return
		else:
			self.generalinfos = self.generalinfomask.search(self.inhtml)
			if self.generalinfos:
				self.Page = 2
				self.showDetails()
				print('############### IMDBquery load IMDBparse_org')
				self.IMDBparse_html()
				return
		
		print('############### IMDBquery read searchResults')
		noresults = True
		self.resultlist = []
		
		if config.plugins.imdb.parsingmethod.value == "json":
			start = self.data.find('"titleResults":{"results":')
			if start != -1:
				noresults = False
				self.jsonData = json.JSONDecoder().raw_decode(self.data, start + 26)[0]
				for result in self.jsonData:
					name_txt = getJsonValues(result,['titleNameText'],"")
					type_txt = getJsonValues(result,['titleTypeText'],"")
					if not type_txt:
						type_txt = getJsonValues(result,['imageType'],"").capitalize()
					release_txt = getJsonValues(result,['titleReleaseText'],"")
					result_txt = name_txt
					if release_txt: result_txt += " (%s)" % release_txt
					if type_txt: result_txt += " (%s)" % type_txt
					self.resultlist.append((str(result_txt),str(result['id'])))
				print("[IMDB] Trefferliste json - results: %s" % len(self.resultlist))
				if len(self.resultlist) == 0: noresults=True
		
		else:
			pos = self.inhtml.find("<table class=\"findList\">")
			if pos > -1:
				#parse findlist
				noresults_text = "No results found for"
				if not re.search('class="findHeader">' + noresults_text, self.inhtml):
					noresults = False
					pos2 = self.inhtml.find("</table>",pos)
					findlist = self.inhtml[pos:pos2]
					searchresultmask = re.compile('<tr class=\"findResult (?:odd|even)\">.*?<td class=\"result_text\"> <a href=\"/title/(tt\d{7,8})/.*?\"\s?>(.*?)</td>', re.S)
					print("[IMDB] Trefferliste 1")
					searchresults = searchresultmask.finditer(findlist)
					self.resultlist = [(self.htmltags.sub('', x.group(2)).strip(), x.group(1)) for x in searchresults]
			else:
				pos = self.inhtml.find('<section data-testid="find-results-section-title"')
				if pos > -1:
					#parse alternate findlist
					pos2 = self.inhtml.find('</section',pos)
					findlist = self.inhtml[pos:pos2]
					noresults_text = "No results found for"
					if not re.search(noresults_text, findlist):
						noresults = False
						searchresultmask = re.compile('<a class=\"ipc-metadata-list-summary-item__t\".*?/title/(tt\d{7,8})/.*?\">(.*?)</a><ul class.*?_tl.base.*?<li.*?<span.*?>(.*?)</span>.*?<li.*?<span.*?>(.*?)</span>', re.S)
						searchresults = list(searchresultmask.finditer(findlist))
						findlist_type = "2a"
						try:
							if len(searchresults) == 0:
								searchresultmask = re.compile('<a class=\"ipc-metadata-list-summary-item__t\".*?/title/(tt\d{7,8})/.*?\">(.*?)</a><ul class.*?_tl.base.*?<li.*?<label.*?>(.*?)</label>.*?<li.*?<label.*?>(.*?)</label>', re.S)
								searchresults = list(searchresultmask.finditer(findlist))
								findlist_type = "2b"
						except:
							import traceback, sys
							traceback.print_exc()
						print("[IMDB] Trefferliste %s - results: %s" % (findlist_type, len(searchresults)))
						searchresults = iter(searchresults)
						self.resultlist = [(self.htmltags.sub('', x.group(2)).strip() + " (" + x.group(3).strip() + ")" + " (" + str(x.group(4)).strip() + ")", x.group(1)) for x in searchresults]

		if noresults == False:
			Len = len(self.resultlist)
			self["menu"].l.setList(self.resultlist)
			if Len == 1:
				self["key_green"].setText("")
				self["statusbar"].setText(_("Loading ...") + " IMDb: %s" % (self.resultlist[0][0]))
				link = self.resultlist[0][1]
				if config.plugins.imdb.parsingmethod.value == "json" and config.plugins.imdb.language.value == "de-de":
					imdb_headers = {'Accept-Language': config.plugins.imdb.language.value}
				else:
					imdb_headers = {}
				fetchurl = "https://www.imdb.com/title/" + link + "/"
				getPage(fetchurl, agent=getRandomUserAgent(), headers=imdb_headers).addCallback(self.IMDBquery, True).addErrback(self.http_failed)
			elif Len > 1:
				self.Page = 1
				self.showMenu()
			else:
				self["statusbar"].setText(_("No IMDb match.") + ' ' + self.eventName)
		else:
			print("[IMDB] getIMDB() No Results found for '%s'" % self.eventName)
			#check for split option in evenName for a second query
			splitpos = self.eventName.find(' (')
			if splitpos < 0:
				splitpos = self.eventName.find('(')
			if splitpos < 0:
				splitpos = self.eventName.find(' - ')
			if splitpos < 0:
				splitpos = self.eventName.find('-')
			if splitpos < 0:
				splitpos = self.eventName.find(': ')

			if splitpos > 0:
				self.eventName = self.eventName[:splitpos].strip()
				self["statusbar"].setText(_("Re-Query IMDb: %s...") % (self.eventName))
				imdb_headers = {}
				fetchurl = "https://www.imdb.com/find/?ref_=nv_sr_fn&q=" + quoteEventName(self.eventName) + "&s=all"
				print("[IMDB] getIMDB() second Downloading Query '%s', '%s'" % (fetchurl, self.eventName))
				getPage(fetchurl, agent=getRandomUserAgent(), headers=imdb_headers).addCallback(self.IMDBquery).addErrback(self.http_failed)
			else:
				self["statusbar"].setText(_("IMDb query failed!") + ' ' + self.eventName)

	def http_failed(self, error):
		text = _("IMDb Download failed")
		text += ": " + str(error)
		print("[IMDB] ",text)
		self["statusbar"].setText(text)

	def IMDBquery2(self, data):
		self.data = data
		self["statusbar"].setText(_("IMDb Re-Download completed"))
		self.html2utf8(data)
		
		if debug_output:
			logfile = open("/media/hdd/imdb.html", "w")
			logfile.write(self.inhtml)
			logfile.close()
			logfile = open("/media/hdd/data.txt", "w")
			logfile.write(self.data)
			logfile.close()
		
		if config.plugins.imdb.parsingmethod.value == "json":
			self.IMDBparse_json()
		else:
			print('############### search generalinfos')
			starttime = time()
			self.generalinfos = self.generalinfomask.search(self.inhtml)
			print("############### duration", time() - starttime)
			if debug_output:
				from pprint import pprint
				pprint(self.generalinfos.groups())
			self.IMDBparse_html()
	
	def IMDBparse_json(self):
		self.Page = 1
		Detailstext = _("No details found.")
		start = self.data.find('pageProps":')
		if start != -1:
			self.jsonData = json.JSONDecoder().raw_decode(self.data, start + 11)[0]
			aboveTheFoldData = self.jsonData['aboveTheFoldData']
			mainColumnData = self.jsonData['mainColumnData']
			translationData = self.jsonData['translationContext']['i18n']['translations']['resources']
			
			if debug_output:
				with open("/media/hdd/mainColumnData.json", "w") as write_file:
					json.dump(mainColumnData, write_file, ensure_ascii=False, indent=4)
				with open("/media/hdd/aboveTheFoldData.json", "w") as write_file:
					json.dump(aboveTheFoldData, write_file, ensure_ascii=False, indent=4)
				with open("/media/hdd/translationData.json", "w") as write_file:
					json.dump(translationData, write_file, ensure_ascii=False, indent=4)
				
				logfile = open("/media/hdd/imdb.html", "w")
				logfile.write(self.inhtml)
				logfile.close()
				
				logfile = open("/media/hdd/data.txt", "w")
				logfile.write(self.data)
				logfile.close()
			
			self["key_yellow"].setText(_("Details"))
			self["statusbar"].setText(_("IMDb Details parsed"))
			
			Titletext = str(getJsonValues(aboveTheFoldData,['titleText', 'text'],""))
			if len(Titletext) > 57:
				Titletext = Titletext[0:54] + "..."
			self.title_setText(Titletext)

			Detailstext = ""
			addnewline = ""
			
			generalinfos = {}
			
			generalinfos["genre"] = list(getJsonValues(genre, ['text'],"") for genre in getJsonValues(aboveTheFoldData,['genres', 'genres'],[]))
			generalinfos["genre_text"] = (_("Genre:"),_("Genres:"))
			
			runtime = getJsonValues(mainColumnData,['runtime', 'seconds'],0)
			if runtime:
				generalinfos["runtime"] = "%s min" % str(runtime // 60)
				generalinfos["runtime_text"] = _("Runtime:")
			
			generalinfos["cert"] = getJsonValues(aboveTheFoldData,['certificate', 'rating'],"")
			generalinfos["cert_text"] = _("Certificate:")
			
			generalinfos["type"] = getJsonValues(aboveTheFoldData,['titleType', 'text'],"")
			generalinfos["type_text"] = _("Type") + ":"
			
			generalinfos["director"] = list(getJsonValues(name, ['name', 'nameText', 'text'],"") for name in getJsonValues(mainColumnData, ['directors', 0, 'credits'],[]))
			generalinfos["director_text"] = (_("Director:"),_("Directors:"))
			
			generalinfos["creator"] = list(getJsonValues(name, ['name', 'nameText', 'text'],"") for name in getJsonValues(mainColumnData, ['creators', 0, 'credits'],[]))
			generalinfos["creator_text"] = (_("Creator:"),_("Creators:"))
			
			generalinfos["writer"] = list(getJsonValues(name, ['name', 'nameText', 'text'],"") + (name['attributes'] and " (" + name['attributes'][0]['text'] + ")" or "") for name in getJsonValues(mainColumnData, ['writers', 0, 'credits'],[]))
			generalinfos["writer_text"] = (_("Writer:"),_("Writers:"))
			
			generalinfos["country"]  = list(getJsonValues(country, ['text'],"") for country in getJsonValues(mainColumnData, ['countriesOfOrigin', 'countries'],[]))
			generalinfos["country_text"] = (_("Country of origin:"),_("Countries of origin:"))
			
			generalinfos["originaltitle"] = str(getJsonValues(mainColumnData, ['originalTitleText', 'text'],""))
			generalinfos["originaltitle_text"] = _("Original title:")
			
			releaseDate = getJsonValues(mainColumnData, ['releaseDate'],{})
			if releaseDate:
				day_val = int(str(releaseDate['day']).replace("None", "0"))
				month_val = int(str(releaseDate['month']).replace("None", "0"))
				year_val = int(str(releaseDate['year']).replace("None", "0"))
				date_txt = ("%02d.%02d.%s" % (day_val, month_val, year_val)).replace("00.","")
				release_country = getJsonValues(releaseDate,["country","text"],"")
				generalinfos["premiere"] = date_txt
				generalinfos["premiere_text"] = _("Release date:")
				if release_country:
					generalinfos["premiere"] = generalinfos["premiere"] + " (%s)" % release_country
			
			generalinfos["alternativ"] = str(getJsonValues(mainColumnData, ['akas', 'edges', 0, 'node', 'text'],""))
			generalinfos["alternativ_text"] = _("Also known as:")
			
			generalinfos["seasons"] =  len(getJsonValues(mainColumnData, ['episodes','seasons'],[]))
			generalinfos["seasons_text"] = _("Seasons:")
			
			generalinfos["episodes"] = int(getJsonValues(mainColumnData, ['episodes','totalEpisodes','total'],0))
			generalinfos["episodes_text"] = _("Episodes:")
			
			if debug_output:
				from pprint import pprint
				pprint(generalinfos)
			
			if generalinfos:
				for category in ("type", "genre", "runtime", "cert", "director", "creator", "writer", "country", "originaltitle", "premiere", "alternativ", "seasons", "episodes"):
					try:
						newline = "\n"
						if not Detailstext: newline = ""
						if generalinfos[category] and isinstance(generalinfos[category],list):
							if len(generalinfos[category]) >1:
								Detailstext += newline + generalinfos[category + "_text"][1] + " " + ", ".join(generalinfos[category]).strip(", ")
							else:
								Detailstext += newline + generalinfos[category + "_text"][0] + " " + ", ".join(generalinfos[category]).strip(", ")
							
						elif generalinfos[category]:
							Detailstext += newline + "%s %s" % (generalinfos[category + "_text"], generalinfos[category])
					
					except (KeyError, TypeError, IndexError):
						print("[IMDb] error generalinfos category", category)
			
			ratingval = getJsonValues(aboveTheFoldData, ['ratingsSummary', 'aggregateRating'],0)
			ratingcount = int(getJsonValues(aboveTheFoldData, ['ratingsSummary', 'voteCount'],0))
			Ratingtext = _("no user rating yet")
			if ratingval:
				count = ''
				if ratingcount: count = " (%s %s)" % ('{0:n}'.format(ratingcount), _("votes"))
				Ratingtext = _("User Rating") + ": %.1f/10" % (ratingval) + count
				self.ratingstars = int(10*round(float(ratingval),1))
				self["stars"].show()
				self["stars"].setValue(self.ratingstars)
			self["starsbg"].show()
			self["ratinglabel"].setText(str(Ratingtext))
			
			castList = list(getJsonValues(mainColumnData, ['cast', 'edges'],[]))
			castText = ""
			for actor in castList:
				actorName = getJsonValues(actor, ['node', 'name', 'nameText', 'text'], "")
				characterName = getJsonValues(actor, ['node', 'characters', 0, 'name'], "")
				episodenumber = getJsonValues(actor, ['node', 'episodeCredits', 'total'], 0)
				startyear = getJsonValues(actor, ['node', 'episodeCredits', 'yearRange', 'year'], "")
				endyear = getJsonValues(actor, ['node', 'episodeCredits', 'yearRange', 'endYear'], "")
				castText += "\n" + actorName + _(" as ") + characterName
				if config.plugins.imdb.showepisodeinfo.value and episodenumber:
					castText += " (%s %s, %s-%s)" % (episodenumber, _("episodes"), startyear, endyear)
			if castText:
				castText = _("Cast:") + " " + castText
			else:
				castText = _("No cast list found in the database.")
			self["castlabel"].setText(str(castText))
			
			outline = str(getJsonValues(aboveTheFoldData, ['plot', 'plotText', 'plainText'],""))
			if outline:
				txt = _("Plot outline:") + "\n" + outline
				self["storylinelabel"].setText(txt)
			else:
				self["storylinelabel"].hide()
			
			posterurl = str(getJsonValues(aboveTheFoldData, ['primaryImage', 'url'],""))
			if posterurl and posterurl.endswith("jpg") > 0:
				self["statusbar"].setText(_("Downloading Poster ..."))
				localfile = "/tmp/poster.jpg"
				print("[IMDB] downloading poster " + posterurl + " to " + localfile)
				download = downloadWithProgress(posterurl,localfile,headers=imdb_headers)
				download.start().addCallback(self.IMDBPoster).addErrback(self.http_failed)
			else:
				self.IMDBPoster("no poster")
		
		self["key_blue"].setText(_("Extra Info"))
		self["extralabel"].hide()
		self["detailslabel"].setText(str(Detailstext))

	def IMDBparse_html(self):
		self.Page = 1
		Detailstext = _("No details found.")
		if self.generalinfos:
			self["key_yellow"].setText(_("Details"))
			self["statusbar"].setText(_("IMDb Details parsed"))
			Titletext = self.generalinfos.group("title").replace(self.NBSP, ' ').strip()
			if len(Titletext) > 57:
				Titletext = Titletext[0:54] + "..."
			self.title_setText(Titletext)

			Detailstext = ""
			addnewline = ""

			_("Genre:"), _("Genres:") # translate tags
			print('###############4 - genreblock')
			genreblock = self.genreblockmask.search(self.inhtml)
			if genreblock:
				info_txt = _("Genre:")
				genrelist = re.split('\|+', self.htmltags.sub('|', genreblock.group("genres")))
				if sum(map(bool, genrelist)) > 1:
					info_txt = _("Genres:")
				genres = ', '.join(genrelist)
				if genres:
					Detailstext += addnewline + info_txt + " " + genres.strip(', ').strip(',')
					addnewline = "\n"

			cert_txt = None
			print('###############4a - certificate')
			self.certificate = self.certificatemask.search(self.inhtml)
			if self.certificate:
				if debug_output:
					from pprint import pprint
					pprint(self.certificate.groups())
				cert_txt = self.certificate.group("cert")
			
			_("Director:"), _("Directors:"), _("Creator:"), _("Creators:"), _("Writer:"), _("Writers:"), _('Episodes:'), _("Runtime:"), _("Release date:"), _("Country of origin:"), _("Countries of origin:"), _("Original title:"), _("Also known as:") # translate tags
			for category in ("director", "creator", "writer", "seasons", "episodes", "runtime", "premiere", "country", "originaltitle", "alternativ"):
				try:

					if self.generalinfos.group(category):
						if category == 'runtime':
							runtime = re.findall('(?:(\d+)\shours?|)\s{0,1}(?:(\d+)\sminutes?|)', self.htmltags.sub('', self.generalinfos.group(category)).replace('<!-- -->',''), re.S)
							print(runtime)
							if runtime:
								if runtime[0][0] and runtime[0][1]:
									runtime = str(60 * int(runtime[0][0]) + int(runtime[0][1]))
									txt = runtime + " min"
								elif runtime[0][0]:
									runtime = str(60 * int(runtime[0][0]))
									txt = runtime + " min"
								elif runtime[0][1]:
									txt = runtime[0][1] + " min"
							else:
								txt = ', '.join(re.split('\|+', self.htmltags.sub('|', self.generalinfos.group(category)).strip('|').replace("\n", ' ')))
						elif category == 'seasons':
							txt = ' '.join(self.htmltags.sub(' ', self.generalinfos.group(category)).replace("\n", ' ').replace('See all', '...').split())
						elif category in ('director', 'creator', 'writer'):
							txt = ', '.join(re.split('\|+', self.htmltags.sub('|', self.generalinfos.group(category).replace('</a><span class="ipc-metadata-list-item__list-content-item--subText">', ' ')).strip('|').replace("\n", ' ')))
						elif category in ("premiere", "country", "originaltitle", "alternativ"):
							txt = ', '.join(re.split('\|+', self.htmltags.sub('|', self.generalinfos.group(category).replace('\n', ' ')).strip('|')))
						else:
							txt = ', '.join(re.split('\|+', self.htmltags.sub('|', self.generalinfos.group(category)).strip('|').replace("\n", ' ')))
						Detailstext += addnewline + _(self.generalinfos.group('g_' + category)+":") + " " + txt
						#insert parental certificate after runtime
						if category == 'runtime' and cert_txt:
							Detailstext += addnewline + _("Certificate:") + " " + cert_txt.replace(self.NBSP, ' ').strip()
						
						if category == 'seasons':
							Detailstext = Detailstext.replace('seasons:', _('Seasons:'))
				except IndexError:
					pass
			
			print('###############5 - rating')
			rating = self.ratingmask.search(self.inhtml)
			Ratingtext = _("no user rating yet")
			if rating:
				ratingval = rating.group("rating")
				if ratingval != '<span id="voteuser"></span>':
					count = ''
					if rating.group("ratingcount"): count = ' (' + rating.group("ratingcount").replace(',','.') + ' ' + _("votes") +')'
					Ratingtext = _("User Rating") + ": " + ratingval + "/10" + count
					self.ratingstars = int(10*round(float(ratingval.replace(',','.')),1))
					self["stars"].show()
					self["stars"].setValue(self.ratingstars)
					self["starsbg"].show()
			self["ratinglabel"].setText(str(Ratingtext))

			print('###############5a - cast')
			castresult = self.castmask.finditer(self.inhtml)
			if castresult:
				Casttext = ""
				for x in castresult:
					Casttext += "\n" + self.htmltags.sub('', x.group('actor'))
					if x.group('character'):
						chartext = self.htmltags.sub(' ', x.group('character').replace('/ ...', '')).replace('\n', ' ').replace(self.NBSP, ' ')
						Casttext += _(" as ") + ' '.join(chartext.split()).replace('…', '')
						try:
							if config.plugins.imdb.showepisodeinfo.value and x.group('episodes'):
								Casttext += ' (' + self.htmltags.sub('', x.group('episodes').replace('eps', _("episodes")).replace(' • ', ', ')).strip() + ')'
						except IndexError:
							pass
				if Casttext:
					Casttext = _("Cast:") + " " + Casttext
				else:
					Casttext = _("No cast list found in the database.")
				self["castlabel"].setText(Casttext)

			print('###############6 - storyline')
			storylineresult = self.storylinemask.search(self.inhtml)
			if storylineresult:
				_("Storyline:") # translate tag
				Storyline = ""
				if storylineresult.group("g_storyline"):
					Storyline = _(storylineresult.group('g_storyline')+":") + "\n"
					if not "Add content advisory" in storylineresult.group('storyline') and not 'data-testid="storyline-parents-guide"' in storylineresult.group('storyline'):
						Storyline = Storyline + re.sub('\s{5,30}',', ', self.htmltags.sub('', storylineresult.group('storyline').replace('\n','').replace('<br>', '\n').replace('<br/>', '\n').replace('<br />','\n').replace('&nbsp;','').replace('&quot;','"').replace('<span class="','')).strip())

				if Storyline == _(storylineresult.group('g_storyline')+":") + "\n":
					self["storylinelabel"].hide()
				else:
					self["storylinelabel"].setText(Storyline)

			print('###############7 - poster')
			posterurl = self.postermask.search(self.inhtml)
			if posterurl and posterurl.group(1).find("jpg") > 0:
				posterurl = posterurl.group(1)
				self["statusbar"].setText(_("Downloading Movie Poster: %s...") % (posterurl))
				localfile = "/tmp/poster.jpg"
				print("[IMDB] downloading poster " + posterurl + " to " + localfile)
				download = downloadWithProgress(posterurl,localfile,headers=imdb_headers)
				download.start().addCallback(self.IMDBPoster).addErrback(self.http_failed)
			else:
				self.IMDBPoster("no poster")
		
		self["key_blue"].setText(_("Extra Info"))
		self["extralabel"].hide()
		self["detailslabel"].setText(str(Detailstext))

	def IMDBPoster(self, string):
		self["statusbar"].setText(_("IMDb Details parsed"))
		if not string:
			filename = "/tmp/poster.jpg"
		else:
			filename = resolveFilename(SCOPE_PLUGINS, "Extensions/IMDb/no_poster.png")
		sc = AVSwitch().getFramebufferScale()
		self.picload.setPara((self["poster"].instance.size().width(), self["poster"].instance.size().height(), sc[0], sc[1], False, 1, "#00000000"))
		self.picload.startDecode(filename)

	def paintPosterPixmapCB(self, picInfo=None):
		ptr = self.picload.getData()
		if ptr != None:
			self["poster"].instance.setPixmap(ptr)
			self["poster"].show()

	def setup(self):
		self.session.open(IMDbSetup)

	def createSummary(self):
		return IMDbLCDScreenV2

class IMDbLCDScreenV2(Screen):
	skin = (
	"""<screen title="IMDB Plugin" position="0,0" size="132,64" id="1">
		<widget name="headline" font="Display;22" halign="center" position="1,3" size="130,22"/>
		<widget source="parent.titlelcd" render="Label" font="Display;16" halign="center" valign="center" position="1,28" size="130,34"/>
	</screen>""",
	"""<screen title="IMDB Plugin" position="0,0" size="96,64" id="2">
		<widget name="headline" font="Display;19" halign="center" position="0,2" size="96,20"/>
		<widget source="parent.titlelcd" render="Label" font="Display;15" halign="center" valign="center" position="0,28" size="96,34"/>
	</screen>""",
	"""<screen title="IMDB Plugin" position="0,0" size="400,240" id="3">
		<ePixmap position="0,0" size="400,240" pixmap="skin_default/display_bg.png" zPosition="-1"/>
		<widget name="headline" font="Display;60" halign="center" position="0,5" size="400,60" transparent="1"/>
		<eLabel backgroundColor="yellow" position="0,75" size="400,2" />
		<widget source="parent.titlelcd" render="Label" font="Display;45" halign="center" valign="center" position="0,85" size="400,135" transparent="1"/>
	</screen>""")

	def __init__(self, session, parent):
		Screen.__init__(self, session, parent)
		self["headline"] = Label(_("IMDb Plugin"))

class IMDbSetup(Screen, ConfigListScreen):
	def __init__(self, session):
		Screen.__init__(self, session)
		self.skinName = "Setup"

		self.setup_title = _("IMDb Setup")
		self.onChangedEntry = []

		self["key_green"] = StaticText(_("OK"))
		self["key_red"] = StaticText(_("Cancel"))
		self["description"] = Label("")

		self["actions"] = ActionMap(["SetupActions"],
			{
				"cancel": self.keyCancel,
				"save": self.keySave,
			}, -2)

		self.list = []
		ConfigListScreen.__init__(self, self.list, session = self.session, on_change = self.changedEntry)
		self.createSetup()
		self.changedEntry()
		self.onLayoutFinish.append(self.layoutFinished)

	def createSetup(self):
		self.list = []
		self.list.append(getConfigListEntry(_("Show in plugin browser"), config.plugins.imdb.showinplugins))
		self.list.append(getConfigListEntry(_("Use original IMDb skin:"), config.plugins.imdb.origskin))
		self.list.append(getConfigListEntry(_("Show episode and year information in cast list"), config.plugins.imdb.showepisodeinfo))
		self.list.append(getConfigListEntry(_("Words / phrases to ignore (comma separated)"), config.plugins.imdb.ignore_tags))
		self.list.append(getConfigListEntry(_("Use parsingmethod"), config.plugins.imdb.parsingmethod))
		if config.plugins.imdb.parsingmethod.value == "json":
			self.list.append(getConfigListEntry(_("IMDb query language"), config.plugins.imdb.language))
		self["config"].list = self.list
		self["config"].l.setList(self.list)

	def layoutFinished(self):
		self.setTitle(_(self.setup_title) + " v" + version)

	def changedEntry(self):
		self.item = self["config"].getCurrent()
		for x in self.onChangedEntry:
			x()
		try:
			if isinstance(self["config"].getCurrent()[1], ConfigYesNo) or isinstance(self["config"].getCurrent()[1], ConfigSelection):
				self.createSetup()
		except:
			pass

	def getCurrentEntry(self):
		return self["config"].getCurrent() and self["config"].getCurrent()[0] or ""

	def getCurrentValue(self):
		return self["config"].getCurrent() and str(self["config"].getCurrent()[1].getText()) or ""

	def getCurrentDescription(self):
		return self["config"].getCurrent() and len(self["config"].getCurrent()) > 2 and self["config"].getCurrent()[2] or ""

	def createSummary(self):
		from Screens.Setup import SetupSummary
		return SetupSummary

	def keySave(self):
		self.saveAll()
		global imdb_headers
		if config.plugins.imdb.parsingmethod.value == "json" and config.plugins.imdb.language.value:
			imdb_headers = {'Accept-Language': config.plugins.imdb.language.value}
		else:
			imdb_headers = {}
		if not config.plugins.imdb.showinplugins.value:
			for plugin in plugins.getPlugins(PluginDescriptor.WHERE_PLUGINMENU):
				if plugin.name == _("IMDb Details"):
					plugins.removePlugin(plugin)
		
		plugins.readPluginList(resolveFilename(SCOPE_PLUGINS))
		self.close()

def eventinfo(session, eventname='', service='', **kwargs):
	if not eventname:
		s = session.nav.getCurrentService()
		if s:
			info = s.info()
			event = info.getEvent(0) # 0 = now, 1 = next
			eventname = event and event.getEventName() or ''
	else:
		eventname = eventname.getEventName()
	session.open(IMDB, eventname)

def main(session, event='', **kwargs):
	session.open(IMDB, event)

pluginlist = PluginDescriptor(name=_("IMDb Details"), description=_("Query details from the Internet Movie Database"), icon="imdb.png", where=PluginDescriptor.WHERE_PLUGINMENU, fnc=main, needsRestart=False)

def Plugins(**kwargs):
	l = [PluginDescriptor(name=_("IMDb Details") + "...", description=_("Query details from the Internet Movie Database"), where=[PluginDescriptor.WHERE_EVENTINFO, PluginDescriptor.WHERE_EPG_SELECTION_SINGLE_BLUE, PluginDescriptor.WHERE_EVENTVIEW], fnc=eventinfo, needsRestart=False,	),]
	if config.plugins.imdb.showinplugins.value:
		l.append(pluginlist)
	return l
